unit spcx ;
{

   Module  : SPCX

   Author  : Hans Otten

   Version : 1.3  3-feb-1991

   Facility: Sixel/PCX/MSX routines

   Purpose : display/convert PCX files


   Routines:

              SCAN_PCX

              Display_PCX

}


interface

  uses

    CNV ;


procedure Scan_PCX (var parse_info : parse_infotype;
                    display_info : display_infotype ;
                    var colormap : colormap_type) ;

procedure Display_PCX(parse_info : parse_infotype ;
                      display_info : display_infotype ;
                      colormap : colormap_type ) ;


implementation

  uses

    CRT, SBUFFER, SGRAPH, SCREATE ;

var

    bits : integer ;
    nr_of_planes   : integer ;
    bytes_per_line : integer ;



  procedure Read_PCX_pixels(parse_info : parse_infotype ;
                            display_info: display_infotype ;
                            var colormap : colormap_type) ;

    { Read/scan PCX file, dependent on display type }

  var

    mask : byte ;
    plane_count,
    bits_per_line,
    nr_of_bits : integer ;
    count,
    bit_count,
    row_count,
    col_count,
    repeat_count : integer ;
    pixel_line : array[1..max_pixel_col] of byte ;
    b : byte ;

    procedure Add_packed_byte(b : integer) ;

    {
      unpack byte in bits to add to pixel_line buffer
    }

    var

      count : integer ;

    begin

      for count := 8 downto 1 do
        begin
          if ((b and 128) = 128)
            then
              pixel_line[bit_count] := (pixel_line[bit_count]) or mask ;
          inc(bit_count) ;
          if bit_count > bits_per_line
            then
              begin
                bit_count := 1 ;
                inc(plane_count) ;
                mask := mask shl 1 ;
                if (plane_count > nr_of_planes)
                  then
                    col_count := parse_info.max_parse_col + 1
              end ;
          b := b shl 1
        end ;

    end ; { Add_packed_byte }


  begin

    { reset to first position of picture bytes }
    Reset_PCX_picture ;
    bits_per_line := bytes_per_line * 8 ;
    bit_count  := 1 ;
    plane_count := 1 ;
    { for all rows in picture }
    for row_count := 1 to parse_info.max_parse_row do
      begin
        for count := 1 to max_pixel_col do
          pixel_line[count] := 0 ;
        mask := 1 ;
        col_count := 1 ;
        bit_count  := 1 ;
        plane_count := 1 ;
        repeat
          b := ord(GetNextByte) ;
          { check for repeat count: 2 msb bits set }
          if (b and $C0) = $C0
            then
              begin
                { mask off 2 msb bits }
                repeat_count := b and $3F ;
                b := ord(GetNextByte) ;
              end
            else
              repeat_count := 1 ;
          count := 1 ;
          repeat
            if (bits = 8) and (nr_of_planes = 1)
              then
                begin
                 pixel_line[col_count] := b ;
                 inc(count) ;
                 inc(col_count)
              end
            else
              begin
                Add_packed_byte(b) ;
                inc(count) ;
              end ;
          until count > repeat_count ;
        until (col_count > parse_info.max_parse_col) ;

        { send all pixels to output }
        for col_count := 1 to parse_info.max_parse_col do
          begin
            if display_info.display <> parse
              then
                Add_pixel(col_count,row_count,pixel_line[col_count])
              else
                begin
                  colormap[pixel_line[col_count]].used := true ;
                  colormap[pixel_line[col_count]].defined := true ;
                end  ;
          end;
        if keypressed
          then
            begin
              if display_info.display = video
                then Close_graph ;
              exit
            end ;
      end ;

      if display_info.display = video
        then
           begin
             writeln(chr(7)) ;
             repeat
             until keypressed ;
           end ;


  end ; { Display_PCX }

procedure Scan_PCX (var parse_info : parse_infotype;
                    display_info : display_infotype ;
                    var colormap : colormap_type) ;

  var

    ch : char ;
    count,
    xl, yl,
    xh,yh      : integer ;


  begin

    { read info from PCX header }
    ch := GetNextByte ;  { manufacturer }
    if ord(ch) <> $0A
      then
         report_error(inv_pcx_vers,fatal, display_info.display) ;
    ch := GetNextByte ;  { version      }
    if (ord(ch) < 1) and (ord(ch) > 5)
      then
        report_error(inv_pcx_vers,fatal, display_info.display) ;
    ch := GetNextByte ;  { encoding     }
    if ord(ch) <> 1
      then
        report_error(inv_pcx_format, fatal, display_info.display) ;
    bits := ord(getNextByte) ;  { bits per sixel }

    if (bits <> 8) and (bits <> 1)
      then
        report_error(inv_pcx_format, fatal, display_info.display) ;

    { low row }
    xl := ord(GetNextByte) ;
    xl := xl + ord(GetNExtByte) * 256 ;
    { low col }
    yl := ord(GetNextByte) ;
    yl := yl + ord(GetNextByte) * 256 ;
    { high row }
    xh := ord(GetNextByte) ;
    xh := xh + ord(GetNextByte) * 256  ;
    { high col }
    yh := ord(GetNextByte) ;
    yh := yh + ord(GetNextByte) * 256 ;
    parse_info.max_parse_col := xh - xl + 1 ;
    parse_info.max_parse_row := yh - yl + 1 ;

    if parse_info.max_parse_col > max_pixel_col
      then
        report_error(toobig, fatal, display_info.display) ;

    { skip HDPI and VDPI }
    for count := 1 to 4 do
      ch := GetNextByte ;

    { read EGA colormap }
    for count := 0 to 15 do
      begin
        colormap[count].defined := true ;
        colormap[count].colors[c_red] :=
                           ord(GetNextByte) ;
        colormap[count].colors[c_green] :=
                           ord(GetNextByte) ;
        colormap[count].colors[c_blue] :=
                           ord(GetNextByte) ;
      end ;

    { skip reserved byte }
    ch := GetNextByte ;
    { get nr of planes }
    nr_of_planes := ord(GetNextByte) ;

    if (bits = 8) and (nr_of_planes = 3)
      then
        report_error(inv_pcx_format, fatal, display_info.display) ;

    { get bytes per line }
    bytes_per_line := ord(GetNextByte) ;
    bytes_per_line := bytes_per_line + 256*ord(GetNextByte) ;

    if parse_info.max_parse_col > max_pixel_col
      then
        report_error(inv_pcx_format, fatal, display_info.display) ;
    if bits = 8
      then
        { read colormap }
        GET_PCX_Colormap(colormap, display_info.display) ;

    Reset_PCX_picture ;

    Read_PCX_Pixels(parse_info,display_info,colormap) ;

    if (bits = 1) and (nr_of_planes = 1)
      then
        { monochrome, ignore colormap if any }
        begin
          parse_info.mono := true ;
          colormap[0].defined := true ;
          colormap[0].colors[c_red] := 0 ;
          colormap[0].colors[c_green] := 0 ;
          colormap[0].colors[c_blue] := 0 ;
          colormap[1].defined := true ;
          colormap[1].colors[c_red] :=   colormax ;
          colormap[1].colors[c_green] := colormax ;
          colormap[1].colors[c_blue] :=  colormax ;
          for count := 2 to 15 do
            begin
              colormap[count].used := false  ;
              colormap[count].defined := false ;
            end ;
        end ;

  end ; { Scan_PCX }


procedure Display_PCX(parse_info : parse_infotype;
                      display_info : display_infotype ;
                      colormap : colormap_type) ;


  begin

    Read_PCX_pixels(parse_info,display_info,colormap) ;

  end ; { Display_PCX }

end. { unit spcx }