
/*
	GUS driver.

	This card rocks for the sound chip emulation.
	
	Voices 0,1,2 are as expected.
	Voice 3 is for noise.
	
	We set up one instrument to be the 'tone', 
	one instrument for periodic noise, and
	one instrument for white noise.
	
	Each of these is generated by this program, and played as 'simple'
	MOD-style instruments.
	
	Something to note -- either due to the Ultrasound library or
	some other reason, we can't play these samples at exactly X Hz.
	This drops a lot of tones in the lower registers, which is very
	undesirable, especially for musical programs.
	
	So, to get around this, we play samples at 64 octaves higher than they
	are.  This way, we get resolution comparable to the 99/4A.
	
	Except for white noise, since its sample is VERY long due to how easy
	it is to hear repeats, we play it at X Hz rather than create a 4096k
	sample.  ;)
*/


#ifdef GUS
#include <unistd.h>
#include <libgus.h>
#include <sys/gus.h>
#include "16bit.h"
#include "emulate.h"
#include "sound.h"

static int  cards;
static int  card;

int
gusdetect()
{
	if ((cards = gus_cards()) <= 0)
		return 0;
	else {
		log("Using GUS for sound... ");
		return 1;
	}
}


/*
	Set up the voices.
	
	Voices 0,1,2 are tones, so are loaded with a tone patch.
	
*/

struct svoice {
	u8         *sample;
	gus_wave_t  wave;
	gus_instrument_t instr;
	gus_layer_t layer;
};

static struct svoice tone, noise0, noise1;

void
voiceinit(struct svoice *v, int usignd, int len, int instr)
{
	memset((void *) v, 0, sizeof(*v));
	v->sample = (void *) malloc(len);
	memset(v->sample, usignd ? 128 : 0, len);

	v->wave.begin.ptr = v->sample;
	v->wave.mode = v->instr.mode = v->layer.mode = GUS_INSTR_SIMPLE;
	v->wave.format = GUS_WAVE_LOOP | usignd;
	v->instr.info.layer = &v->layer;
	v->instr.number.instrument = instr;
	v->layer.wave = &v->wave;

	v->wave.size = len;
	v->wave.loop_end = len << 4;
}

#define	HZMUL 64
void
maketone()
{
	s8         *ptr;
	int         x;

/*	voiceinit(&tone,GUS_WAVE_UNSIGNED,64,0);
	
	ptr=tone.sample;
	*ptr++=0;
	*ptr++=128;
	*ptr++=128;
	*ptr++=128;
	*ptr++=128;*/

	voiceinit(&tone, 0, 64, 0);
	ptr = tone.sample;
	memset(ptr, -128, 64);

	*ptr++ = 127;
	*ptr++ = 127;
	*ptr++ = 127;
	*ptr++ = 127;
	*ptr++ = 127;
	*ptr++ = 127;
	*ptr++ = 127;
	*ptr++ = 127;

	if (gus_memory_alloc(&tone.instr) < 0)
		fatal("Couldn't allocate voice #0");
}

#define N0MUL 64
void
makenoise0()
{
	s8         *ptr;
	int         x;

	voiceinit(&noise0, 0, 64 * 16, 1);

	ptr = noise0.sample;
	for (x = 0; x < 64; x++)
		*ptr++ = 127;

	if (gus_memory_alloc(&noise0.instr) < 0)
		fatal("Couldn't allocate noise #0");
}

#define N1MUL 1
void
makenoise1()
{
	s8         *ptr;
	int         x, y;
	unsigned short r0, r1;

	voiceinit(&noise1, 0, 65536, 2);

	ptr = noise1.sample;
	r0 = r1 = 0xffff;
	while (ptr < noise1.sample + 65536) {
		*ptr++ = (r0 & 1) ? 127 : -128;

		r0 = (r0 << 1) | (r0 >> 15);
		r0 ^= r1;
		if ((r1 += r0) == 0)
			r1 = 0xffff;
	}

	if (gus_memory_alloc(&noise1.instr) < 0)
		fatal("Couldn't allocate noise #1");
}

void
setupvoices()
{
	int         x;
	unsigned short cx, bp;

	/*  No 'gradual fadeins' pleez  */
	gus_ultraclick_t click;

	click.volume_ramp = 0;
	gus_set_ultraclick(&click);

	maketone();
	makenoise0();
	makenoise1();

	gus_timer_start();
}

#define BUFFERSIZE (200*25)
static u8  *buffer;
static int  buflen;
static int  dsp;

void
setupspeech()
{
	int         stereo = 0;
	int         speed = 8000;

	buffer = (void *) malloc(BUFFERSIZE);
	buflen = 0;

	dsp = open("/dev/dsp", O_WRONLY);
	if (dsp == -1) {
		log("/dev/dsp not available for speech... ");
		return;
	}

	if (ioctl(dsp, SNDCTL_DSP_STEREO, &stereo) == -1 ||
		ioctl(dsp, SNDCTL_DSP_SPEED, &speed) == -1) {
		log("sound driver doesn't support mono/8000 Hz... ");
		return;
	}

	log("got GUS for speech... ");
	features |= FE_SPEECH;
}

int
gusattach()
{
	for (card = 0; card < cards; card++) {
		if (gus_open(card, 512, 0) == 0) {
			gus_queue_write_set_size(512);
			gus_memory_reset(GUS_DOWNLOAD_MODE_NORMAL);
			gus_reset(4, 0);
			gus_select(card);
			setupvoices();
			setupspeech();

			log("done, card #%d\n", card);
			return 1;
		}
	}
	log("failed, no GUS cards available\n");
	return 0;
}

int
gusdetach()
{
	gus_close(card);
	return 1;
}

void
gusspeakeron()
{
//  gus_reset_engine_only();
	gus_do_voice_start(0, 0, 0, 16384, 8192);
	gus_do_voice_start(1, 0, 0, 16384, 8192);
	gus_do_voice_start(2, 0, 0, 16384, 8192);

//  gus_do_voice_start(3,1,0,16384,8192);
//  gus_do_voice_start(4,2,0,16384,8192);
}

void
gusspeakeroff()
{
	int         x;

	for (x = 0; x < 4; x++)
		gus_do_voice_control(x, 3);
//  gus_reset_engine_only();
}

int
getvol(u8 voice)
{
	return (15 - voices[voice].volume) * 512;
}

void
gusvoice(u8 voice)
{

/*	if (voice==2 && (voices[3].stype&3)==3)
		gus_do_voice_volume(voice,0);*/
//  log("Voice %d, %d per, %d Hz\n",voice,voices[voice].period,voices[voice].hertz);
	gus_do_voice_frequency(voice, voices[voice].hertz * HZMUL);
//  gus_do_wait(1);
	gus_do_flush();
//  gus_do_stop();
}

void
gusvoicevol(u8 voice)
{
	gus_do_voice_volume(voice, getvol(voice));
//  gus_do_wait(1);
//  gus_do_stop();      
	gus_do_flush();
}

u8          laststype;
void
gusnoise()
{
	int         inst = (voices[3].stype & 4) != 0;
	int         hz = voices[3].hertz * (inst ? N1MUL : N0MUL);

	if ((voices[3].stype & 7) != (laststype & 7)) {
		gus_do_voice_start(3, inst + 1, hz, getvol(3), 8192);
	} else
		gus_do_voice_frequency(3, hz);

//  log("GUS noise %x, %d hz\n",voices[3].stype,voices[3].hertz);

/*	gus_do_voice_stop(3,0);
	gus_do_voice_start(3,1,voices[3].hertz*NMUL,getvol(3),8192);
	gus_do_voice_frequency(3,voices[3].hertz*NMUL);*/

	gus_do_flush();

	laststype = voices[3].stype;
}

void
gusnoisevol()
{
	gus_do_voice_volume(3, getvol(3));
	gus_do_flush();
}

void
gustoggle()
{
	gus_do_flush();
}

void
gusspeech(u8 * data, int len)
{
//  log("GUS speech:  %d bytes\n",len);
	if (buflen >= BUFFERSIZE || len == 0) {
		write(dsp, buffer, buflen);
		buflen = 0;
	}

	if (len) {
		memcpy(buffer + buflen, data, len);
		buflen += len;
	}
//  gus_pcm_write(data,len);
//  gus_pcm_sync();
}

static void
nullhandler()
{
}

struct sounddevice sound_gus = {
	gusdetect, gusattach, gusdetach,
	gusvoice, gusvoicevol,
	gusnoise, gusnoisevol, gustoggle,
	gusspeakeron, gusspeakeroff,
	gusspeech
};

#endif
