// tape.c
#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>
#include <commdlg.h>
#include <cderr.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "resource.h"
#include "msxcore.h"
#include "dialogs.h"
#include "tape.h"
#include "msx.h"
#include "mem.h"

char				g_szTapeFile[_MAX_PATH] ;
static BYTE		_huge* hpExp, _huge* hpExpPtr ;
static TAPEHEADER	*g_ptphdr ;
static const char	BASED_CODE g_szTapeFilter[] = "Tape Files (*.tap)\0*.tap\0All Files (*.*)\0*.*\0", 
				BASED_CODE g_szTapeDefExt[] = "tap", 
				BASED_CODE szNoFile[] = "<< none: select File >>", 
				BASED_CODE szNoAddFile[] = "<< none: select Add File >>" ;
extern HWND		g_hwndMain ;
extern HINSTANCE	g_hInstance ;
extern const char	g_szAppName[] ;
extern BOOL		g_bLoading, g_bSaving ;
extern BYTE _huge*	g_hpLoadData, _huge* g_hpSaveData ;
extern DWORD		g_dwSizeSave ;

/*
 * A tape file has the following structure:
 *  first a TAPE chunk containing several data chunks
 *  and then a INFO chunk containing info about the data chunks
 */

static int NEAR FileErr (HWND hwnd, UINT nErr, LPSTR lpszFile, UINT uStyle)
	{
	char		szBuf[256], szBuf2[_MAX_PATH] ;
	
	LoadString (g_hInstance, nErr, szBuf, 256) ;
	wsprintf (szBuf2, szBuf, lpszFile) ;
	return MessageBox (hwnd, szBuf2, g_szAppName, uStyle) ;
	}

static TAPEHEADER* NEAR CreateTape (HWND hwnd, LPSTR lpszFileName)
	{
	HMMIO		hmmio ;
	HFILE		hFile ;
	TAPEHEADER	*ptphdr ;
	MMCKINFO		ckinfo ;
	OFSTRUCT		of ;
	
	hFile = OpenFile (lpszFileName, &of, OF_WRITE | OF_CREATE) ;
	if (hFile == HFILE_ERROR)
		{
		switch (of.nErrCode)
			{
			case 0x0002:
			case 0x0003:
				FileErr (hwnd, IDS_FILEWRITEPATH, lpszFileName, MB_ICONEXCLAMATION) ;
				break ;
			
			case 0x0005:
				FileErr (hwnd, IDS_FILEWRITEACCESS, lpszFileName, MB_ICONEXCLAMATION) ;
				break ;
			
			default:
				FileErr (hwnd, IDS_FILEWRITEUNEXPECT, lpszFileName, MB_ICONEXCLAMATION) ;
				break ;
			}
		
		return NULL ;
		}
	
	_lclose (hFile) ;
	
	hmmio = mmioOpen (lpszFileName, NULL, MMIO_WRITE | MMIO_ALLOCBUF | MMIO_CREATE) ;
	if (hmmio == NULL)
		{
		FileErr (hwnd, IDS_FILEWRITEUNEXPECT, lpszFileName, MB_ICONEXCLAMATION) ;
		
		return NULL ;
		}
	
	ckinfo.fccType = mmioFOURCC ('T', 'A', 'P', 'E') ;
	if (mmioCreateChunk (hmmio, &ckinfo, MMIO_CREATELIST) || 
		mmioAscend (hmmio, &ckinfo, 0) )
		goto err1 ;
	
	if (!NewLocal ((void**)&ptphdr, LPTR, sizeof (TAPEHEADER) ) )
		goto err1 ;
	ptphdr->wVersion = 0x100 ;
	
	ckinfo.ckid = mmioFOURCC ('I', 'N', 'F', 'O') ;
	if (mmioCreateChunk (hmmio, &ckinfo, 0) || 
		mmioWrite (hmmio, (HPSTR)ptphdr, (long)sizeof (TAPEHEADER) ) 
			!= (long)sizeof (TAPEHEADER) || 
		mmioAscend (hmmio, &ckinfo, 0) )
		goto err2 ;
	
	mmioClose (hmmio, 0) ;
	
	return ptphdr ;
err2:
	FreeLocal ((void**)&ptphdr) ;
err1:
	mmioOpen (lpszFileName, NULL, MMIO_DELETE) ;
	
	FileErr (hwnd, IDS_FILEWRITEFULL, lpszFileName, MB_ICONEXCLAMATION) ;
	
	return NULL ;
	}

static TAPEHEADER* NEAR GetTapeInfo (HWND hwnd, LPSTR lpszFileName)
	{
	HMMIO		hmmio ;
	TAPEHEADER	*ptphdr ;
	MMCKINFO		ckinfo ;
	OFSTRUCT		of ;
	HFILE		hFile ;
	
	hFile = OpenFile (lpszFileName, &of, OF_READ) ;
	if (hFile == HFILE_ERROR)
		{
		switch (of.nErrCode)
			{
			case 0x0002:	// file not found
				if (FileErr (hwnd, IDS_FILENOTEXIST, lpszFileName, MB_ICONQUESTION | MB_OKCANCEL) == IDOK)
					return CreateTape (hwnd, lpszFileName) ;
				
				break ;
			
			case 0x0003:	// path not found
				FileErr (hwnd, IDS_FILEREADPATH, lpszFileName, MB_ICONEXCLAMATION) ;
				break ;
			
			case 0x0005:	// access denied
				FileErr (hwnd, IDS_FILEREADACCESS, lpszFileName, MB_ICONEXCLAMATION) ;
				break ;
			
			default:		// any other
				FileErr (hwnd, IDS_FILEREADUNEXPECT, lpszFileName, MB_ICONEXCLAMATION) ;
				break ;
			}
		
		return NULL ;
		}
	
	_lclose (hFile) ;
	
	hmmio = mmioOpen (lpszFileName, NULL, MMIO_READ | MMIO_ALLOCBUF) ;
	if (hmmio == NULL)
		{
		FileErr (hwnd, IDS_FILEREADUNEXPECT, lpszFileName, MB_ICONEXCLAMATION) ;
		
		return NULL ;
		}
	
	ckinfo.ckid = mmioFOURCC ('I', 'N', 'F', 'O') ;
	if (mmioDescend (hmmio, &ckinfo, NULL, MMIO_FINDCHUNK) )
		goto err1 ;
	
	if (!NewLocal ((void**)&ptphdr, LMEM_FIXED, (UINT)ckinfo.cksize) )
		goto err1 ;
	
	if (mmioRead (hmmio, (HPSTR)ptphdr, (long)LOWORD (ckinfo.cksize) ) 
			!= (long)LOWORD (ckinfo.cksize) )
		goto err2 ;
	
	if (ptphdr->wVersion != 0x100 || 
		(ptphdr->wTapeRecordings * sizeof (TAPEINFO) + sizeof (TAPEHEADER) ) 
			< LOWORD (ckinfo.cksize) )
		goto err2 ;
	
	if (mmioAscend (hmmio, &ckinfo, 0) )
		goto err2 ;
	
	mmioClose (hmmio, 0) ;
	
	return ptphdr ;
	
err2:
	FreeLocal ((void**)&ptphdr) ;
err1:
	mmioClose (hmmio, 0) ;
	FileErr (hwnd, IDS_FILETAPE, lpszFileName, MB_ICONEXCLAMATION) ;
	
	return NULL ;
	}

static BYTE _huge* NEAR TapeLoadChunk (HWND hwnd, LPSTR lpszFileName, FOURCC fcc, DWORD* pdwSize)
	{
	BYTE _huge*	hpData, _huge* hpTemp ;
	HMMIO		hmmio ;
	MMCKINFO		ckinfoLIST, ckinfo ;
	DWORD		dw ;
	
	hmmio = mmioOpen (lpszFileName, NULL, MMIO_READ | MMIO_ALLOCBUF) ;
	if (hmmio == NULL)
		{
		FileErr (hwnd, IDS_FILEREADUNEXPECT, lpszFileName, MB_ICONEXCLAMATION) ;
		
		return NULL ;
		}
	
	if (mmioDescend (hmmio, &ckinfoLIST, NULL, 0) || 
		ckinfoLIST.ckid != FOURCC_LIST || 
		ckinfoLIST.fccType != mmioFOURCC ('T', 'A', 'P', 'E') )
		goto err1 ;
	
	ckinfo.ckid = fcc ;
	if (mmioDescend (hmmio, &ckinfo, &ckinfoLIST, MMIO_FINDCHUNK) )
		goto err1 ;
	
	if (!NewGlobal ((void FAR**)&hpData, GMEM_MOVEABLE, ckinfo.cksize) )
		goto err1 ;
	
	if (mmioRead (hmmio, hpData, ckinfo.cksize) != 
		(long) ckinfo.cksize)
		goto err2 ;
	
	// is this nessecary ???
	if (mmioAscend (hmmio, &ckinfo, 0) )
		goto err2 ;
	
	if (mmioAscend (hmmio, &ckinfoLIST, 0) )
		goto err2 ;
	
	mmioClose (hmmio, 0) ;
	
	if (pdwSize != NULL)
		*pdwSize = ckinfo.cksize ;
	
	// check chunk
	hpTemp = hpData ;
	while (1)
		{
		dw = *(DWORD _huge*)hpTemp ;
		if (dw == -1L)
			break ;
		hpTemp += dw + 4L ;
		if ((DWORD)(hpTemp - hpData) > ckinfo.cksize)
			{
			FreeGlobal ((void FAR**)&hpData) ;
			DeActivate () ;
			MessageID (hwnd, IDS_CHUNKCORRUPT, MB_OK | MB_ICONEXCLAMATION) ;
			
			return NULL ;
			}
		}
	
	return hpData;
	
err2:
	FreeGlobal ((void FAR**)&hpData) ;
err1:
	DeActivate () ;
	MessageID (hwnd, IDS_ERRLOADCHUNK, MB_OK | MB_ICONEXCLAMATION) ;
	
	return NULL ;
	}

static FOURCC NEAR TapeCreateChunk (HWND hwnd, LPSTR lpszFileName, BYTE _huge* hpData, DWORD dwSize)
	{
	unsigned char	*pch ;
	int			i ;
	MMCKINFO		ckinfoLIST, ckinfo ;
	HMMIO		hmmio ;
	
	hmmio = mmioOpen (lpszFileName, NULL, MMIO_ALLOCBUF | MMIO_READWRITE) ;
	if (hmmio == NULL)
		{
		FileErr (hwnd, IDS_FILEWRITEUNEXPECT, lpszFileName, MB_ICONEXCLAMATION) ;
		
		return (FOURCC) 0L ;
		}
	
	if (mmioDescend (hmmio, &ckinfoLIST, NULL, 0) || 
		ckinfoLIST.ckid != FOURCC_LIST || 
		ckinfoLIST.fccType != mmioFOURCC ('T', 'A', 'P', 'E') )
		goto err1 ;
	
	if (mmioAscend (hmmio, &ckinfoLIST, 0) )
		goto err1 ;
	
	ckinfo.ckid = (FOURCC) GetTickCount () ;
	pch = (unsigned char*)&ckinfo.ckid ;
	for (i=0;i<4;i++)
		if (pch[i] < ' ')pch[i] += ' ' ;
	
	if (mmioCreateChunk (hmmio, &ckinfo, 0) ||
		mmioWrite (hmmio, hpData, (long) dwSize) != (long) dwSize )
		goto err1 ;
	
	if (mmioAscend (hmmio, &ckinfo, 0) )
		goto err1 ;
	
	// thingie already ascended
	ckinfoLIST.dwFlags |= MMIO_DIRTY ;
	if (mmioAscend (hmmio, &ckinfoLIST, 0) )
		goto err1 ;
	
	mmioClose (hmmio, 0) ;
	
	return ckinfo.ckid ;
	
err1:
	mmioClose (hmmio, 0) ;
	
	DeActivate () ;
	MessageID (hwnd, IDS_ERRNEWCHUNK, MB_OK | MB_ICONEXCLAMATION) ;
	
	return (FOURCC) 0L ;
	}

static BOOL NEAR SetTapeInfo (HWND hwnd, LPSTR lpszFileName, TAPEHEADER* ptphdr)
	{
	MMCKINFO		ckinfoLIST, ckinfo ;
	HMMIO		hmmio ;
	long			lSize ;
	int			n, i ;
	
	hmmio = mmioOpen (lpszFileName, NULL, MMIO_ALLOCBUF | MMIO_READWRITE) ;
	if (hmmio == NULL)
		{
		FileErr (hwnd, IDS_FILEWRITEUNEXPECT, lpszFileName, MB_ICONEXCLAMATION) ;
		
		return FALSE ;
		}
	
	if (mmioDescend (hmmio, &ckinfoLIST, NULL, 0) || 
		ckinfoLIST.ckid != FOURCC_LIST || 
		ckinfoLIST.fccType != mmioFOURCC ('T', 'A', 'P', 'E') )
		goto err1 ;
	
	if (mmioAscend (hmmio, &ckinfoLIST, 0) )
		goto err1 ;
	
	for (n=0;n<(int)ptphdr->wTapeRecordings;n++)
		{
		for (i=0;i<32;i++)
			{
			if (ptphdr->tapinf[n].szTitle[i] == 0)
				while (++i<32)
					ptphdr->tapinf[n].szTitle[i] = 0 ;
			}
		
		for (i=0;i<10;i++)
			{
			if (ptphdr->tapinf[n].szType[i] == 0)
				while (++i<10)
					ptphdr->tapinf[n].szType[i] = 0 ;
			}
		}
	
	ckinfo.ckid = mmioFOURCC ('I', 'N', 'F', 'O') ;
	lSize = (long) (sizeof (TAPEHEADER) + ptphdr->wTapeRecordings * sizeof (TAPEINFO) ) ;
	if (mmioCreateChunk (hmmio, &ckinfo, 0) ||
		mmioWrite (hmmio, (HPSTR)ptphdr, lSize) != lSize)
		goto err1 ;
	
	if (mmioAscend (hmmio, &ckinfo, 0) )
		goto err1 ;
	
	mmioClose (hmmio, 0) ;
	
	return TRUE ;
	
err1:
	mmioClose (hmmio, 0) ;
	
	DeActivate () ;
	MessageID (hwnd, IDS_ERRNEWCHUNK, MB_OK | MB_ICONEXCLAMATION) ;
	
	return FALSE ;
	}

static BOOL NEAR TapeRemoveChunk (HWND hwnd, LPSTR lpszFileName, FOURCC fccRemove)
	{
	char			szTemp[_MAX_PATH] ;
	HMMIO		hmmioTemp, hmmio ;
	MMCKINFO		ckinfo, ckinfoLIST, ckinfotemp, ckinfotempLIST ;
	HPSTR		hpData ;
	
	_fstrcpy (szTemp, lpszFileName) ;
	strcpy (szTemp + strlen (szTemp) - 3, "$$$") ;
	
	hmmio = mmioOpen (lpszFileName, NULL, MMIO_READ) ;
	if (hmmio == NULL)
		{
		FileErr (hwnd, IDS_FILEWRITEFULL, lpszFileName, MB_ICONEXCLAMATION) ;
		
		return FALSE ;
		}
	
	hmmioTemp = mmioOpen (szTemp, NULL, MMIO_CREATE | MMIO_WRITE | MMIO_ALLOCBUF) ;
	if (hmmioTemp == NULL)
		{
		mmioClose (hmmio, 0) ;
		
		FileErr (hwnd, IDS_FILEWRITEFULL, lpszFileName, MB_ICONEXCLAMATION) ;
		
		return FALSE ;
		}
	
	if (mmioDescend (hmmio, &ckinfoLIST, NULL, 0) || 
		ckinfoLIST.ckid != FOURCC_LIST || 
		ckinfoLIST.fccType != mmioFOURCC ('T', 'A', 'P', 'E') )
			goto err1 ;
	
	ckinfotempLIST.fccType = mmioFOURCC ('T', 'A', 'P', 'E') ;
	if (mmioCreateChunk (hmmioTemp, &ckinfotempLIST, MMIO_CREATELIST) )
		goto err1 ;
	
	while (1)
		{
		if (mmioDescend (hmmio, &ckinfo, &ckinfoLIST, 0) )
			break ;
		
		if (ckinfo.ckid == fccRemove)
			{
			if (mmioAscend (hmmio, &ckinfo, 0) )
				goto err1 ;
			
			continue ;
			}
		
		if (!NewGlobal ((void FAR**)&hpData, GMEM_MOVEABLE, ckinfo.cksize) )
			goto err1 ;
		
		if (mmioRead (hmmio, hpData, (long)ckinfo.cksize) != (long)ckinfo.cksize || 
			mmioAscend (hmmio, &ckinfo, 0) )
			goto err2 ;
		
		ckinfotemp.ckid = ckinfo.ckid ;
		if (mmioCreateChunk (hmmioTemp, &ckinfotemp, 0) || 
			mmioWrite (hmmioTemp, hpData, (long)ckinfo.cksize) != (long)ckinfo.cksize || 
			mmioAscend (hmmioTemp, &ckinfotemp, 0) )
				goto err2 ;
		
		FreeGlobal ((void FAR**)&hpData) ;
		}
	
	if (mmioAscend (hmmioTemp, &ckinfotempLIST, 0) )
		goto err1 ;
	
	if (mmioAscend (hmmio, &ckinfoLIST, 0) )
		goto err1 ;
	
	mmioClose (hmmio, 0) ;
	mmioClose (hmmioTemp, 0) ;
	
	if (!mmioOpen (lpszFileName, NULL, MMIO_DELETE) || 
		mmioRename (szTemp, lpszFileName, NULL, 0L) )
		{
		FileErr (hwnd, IDS_FILEWRITEFULL, lpszFileName, MB_ICONEXCLAMATION) ;
		
		return FALSE ;
		}
	
	return TRUE ;

err2:
	FreeGlobal ((void FAR**)&hpData) ;
err1:
	mmioClose (hmmio, 0) ;
	mmioClose (hmmioTemp, 0) ;
	
	mmioOpen (szTemp, NULL, MMIO_DELETE) ;
	
	FileErr (hwnd, IDS_FILEWRITEFULL, lpszFileName, MB_ICONEXCLAMATION) ;
	
	return FALSE ;
	}

// dialog procedures
static int		nSelect ;
static TAPEINFO	*ptapinf ;

static BOOL NEAR TapeOpenDlg (HWND hwnd)
	{
	char			szBuf[_MAX_PATH] ;
	static const char BASED_CODE	szTitle[] = "Open Tape File" ;
	OPENFILENAME	ofn ;
	TAPEHEADER	*ptphdr ;
	
	szBuf[0] = '\0' ;
	memset (&ofn, 0, sizeof (OPENFILENAME) ) ;
	ofn.lStructSize  = sizeof (OPENFILENAME) ;
	ofn.hwndOwner    = hwnd ;
	ofn.lpstrFilter  = g_szTapeFilter ;
	ofn.lpstrFile    = szBuf ;
	ofn.nMaxFile     = sizeof (szBuf) ;
	ofn.lpstrTitle   = szTitle ;
	ofn.Flags        = OFN_HIDEREADONLY ;
	ofn.lpstrDefExt  = g_szTapeDefExt ;
	
	if (GetOpenFileName (&ofn) )
		{
		AnsiUpper (szBuf) ;
		ptphdr = GetTapeInfo (hwnd, szBuf) ;
		if (ptphdr != NULL)
			{
			if (g_szTapeFile[0] && g_ptphdr != NULL)
				FreeLocal ((void**)&g_ptphdr) ;
			g_ptphdr = ptphdr ;
			strcpy (g_szTapeFile, szBuf) ;
			
			return TRUE ;
			}
		}
	
	return FALSE ;
	}

// saving from the emulator
static void NEAR SetFonts (HWND hDlg)
	{
	HFONT	hFont ;
	LOGFONT	lFont ;

	hFont = (HFONT) SendMessage (hDlg, WM_GETFONT, 0, 0L) ;
	if (hFont != NULL)
		if (GetObject (hFont, sizeof (LOGFONT), &lFont) )
			{
			lFont.lfWeight = FW_NORMAL ;
			hFont = CreateFontIndirect (&lFont)  ;
			if (hFont != NULL)
				{
				SendDlgItemMessage (hDlg, IDC_CURTAPEFILE, WM_SETFONT, 
					(WPARAM)hFont, 0L) ;
				SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, WM_SETFONT, 
					(WPARAM)hFont, 0L) ;
				SendDlgItemMessage (hDlg, IDC_IMPTAPFILE, WM_SETFONT, 
					(WPARAM)hFont, 0L) ;
				}
			}
	
	}

static void NEAR SetCurTapeFile (HWND hDlg)
	{
	SetWindowText (GetDlgItem (hDlg, IDC_CURTAPEFILE), (g_szTapeFile[0] ?
		(LPCSTR)g_szTapeFile : (LPCSTR)szNoFile) ) ;
	EnableWindow (GetDlgItem (hDlg, IDOK), g_szTapeFile[0] != 0) ;
	}

static int NEAR GetListIndex (char *szBuf, TAPEHEADER* ptphdr)
	{
	char		szBuf2[256] ;
	int		i ;
	
	for (i=0;i<(int)ptphdr->wTapeRecordings;i++)
		{
		wsprintf (szBuf2, "%s\t%s", 
			(LPSTR) ptphdr->tapinf[i].szTitle, 
			(LPSTR) ptphdr->tapinf[i].szType) ;
		
		if (!strcmp (szBuf2, szBuf) )
			return i ;
		}
	
	assert (0) ;
	
	return -1 ;
	}

BOOL FAR PASCAL _export TapeSaveDlgProc (HWND hDlg, UINT message, UINT wParam, 
												LONG lParam)
	{
	int		i ;
	char		szBuf[256] ;
	
	switch (message)
		{
		case WM_INITDIALOG :
			SetCurTapeFile (hDlg) ;
			SetFonts (hDlg) ;
			SetWindowText (GetDlgItem (hDlg, IDC_TITLEEDIT), ptapinf->szTitle) ;
			SetWindowText (GetDlgItem (hDlg, IDC_TYPEEDIT), ptapinf->szType) ;
			SetFonts (hDlg) ;
			
			return TRUE ;
		
		case WM_COMMAND :
			switch (wParam)
				{
				case IDC_TITLEEDIT:
					if (HIWORD (lParam) == EN_CHANGE)
						{
						GetWindowText ((HWND)LOWORD (lParam), szBuf, 256) ;
						if (strlen (szBuf) > 31)
							{
							MessageID (hDlg, IDS_MSGTITLEMAX, MB_OK | MB_ICONEXCLAMATION) ;
							szBuf[31] = '\0' ;
							SetWindowText ((HWND)LOWORD (lParam), szBuf) ;
							}
						strcpy (ptapinf->szTitle, szBuf) ;
						}
					
					return TRUE ;
				
				case IDC_TYPEEDIT:
					if (HIWORD (lParam) == EN_CHANGE)
						{
						GetWindowText ((HWND)LOWORD (lParam), szBuf, 256) ;
						if (strlen (szBuf) > 9)
							{
							MessageID (hDlg, IDS_MSGTYPEMAX, MB_OK | MB_ICONEXCLAMATION) ;
							szBuf[9] = '\0' ;
							SetWindowText ((HWND)LOWORD (lParam), szBuf) ;
							}
						strcpy (ptapinf->szType, szBuf) ;
						}
					
					return TRUE ;
				
				case IDOK :
					if (!ptapinf->szTitle[0] || !ptapinf->szType[0])
						{
						MessageID (hDlg, IDS_MSGNOTEXT, MB_OK | MB_ICONEXCLAMATION) ;
						return TRUE ;
						}
					
					for (i=0;i<(int)g_ptphdr->wTapeRecordings;i++)
						{
						if (!strcmp (ptapinf->szTitle, g_ptphdr->tapinf[i].szTitle) &&
							!strcmp (ptapinf->szType, g_ptphdr->tapinf[i].szType) )
							{
							MessageID (hDlg, IDS_MSGEXIST, MB_OK | MB_ICONEXCLAMATION) ;
							
							return TRUE ;
							}
						}
					
				case IDCANCEL :
					EndDialog (hDlg, wParam == IDOK) ;
					return TRUE ;
				
				case IDC_FILE:
					if (TapeOpenDlg (hDlg) )
						SetCurTapeFile (hDlg) ;
					
					return TRUE ;
				}
			
			break ;
		}
	
	return FALSE ;
	}

static BOOL NEAR GetNameAndType (BYTE _huge* hpByte, TAPEINFO *ptapinf)
	{
	static const char 
		BASED_CODE	szBasic[] = "basic", 
		BASED_CODE	szAscii[] = "ascii", 
		BASED_CODE	szBinary[] = "binary", 
		BASED_CODE	szMaster[] = "Game Mstr", 
		BASED_CODE	szValley[] = "KV2 stage", 
		BASED_CODE	szROM[] = "GetROM", 
		BASED_CODE	szUnknown[] = "<unknown>" ;
	BYTE		byte ;
	int		i ;
	
	switch (*(DWORD _huge*)hpByte)
		{
		case 0x10L:
			hmemcpy (ptapinf->szTitle, hpByte + 14, 6) ;
			MsxToAnsiBuf (ptapinf->szTitle, 6) ;
			DeleteSpaces (ptapinf->szTitle, 6) ;
			byte = hpByte[4] ;
			for (i=5;i<13;i++)
				if (hpByte[i] != byte)
					{
					byte = 0 ;
					break ;
					}
			
			switch (byte)
				{
				case 0xd3:
					_fstrcpy (ptapinf->szType, szBasic) ;
					
					return TRUE ;
				
				case 0xd0:
					_fstrcpy (ptapinf->szType, szBinary) ;
					
					return TRUE ;
				
				case 0xea:
					_fstrcpy (ptapinf->szType, szAscii) ;
					
					return TRUE ;
				}
			
			break ;
		
		case 0x18L:
			// king's valley 2 stage file ?
			// ROM saving from GetROM ?
			byte = hpByte[4] ;
			for (i=5;i<0x14;i++)
				if (hpByte[i] != byte)
					{
					byte = 0 ;
					break ;
					}
			
			if (byte == 0xaa)
				{
				_fstrcpy (ptapinf->szType, szValley) ;
				hmemcpy (ptapinf->szTitle, hpByte + 0x14, 8) ;
				MsxToAnsiBuf (ptapinf->szTitle, 8) ;
				DeleteSpaces (ptapinf->szTitle, 8) ;
				
				return TRUE ;
				}
			
			if (byte == 0x55)
				{
				_fstrcpy (ptapinf->szType, szROM) ;
				hmemcpy (ptapinf->szTitle, hpByte + 0x14, 8) ;
				MsxToAnsiBuf (ptapinf->szTitle, 8) ;
				DeleteSpaces (ptapinf->szTitle, 8) ;
				
				return TRUE ;
				}
			
			break ;
		
		case 0x12L:
			// game master file ?
			byte = hpByte[4] ;
			for (i=5;i<0xe;i++)
				if (hpByte[i] != byte)
					{
					byte = 0 ;
					break ;
					}
			
			if (byte == 0xb)
				{
				_fstrcpy (ptapinf->szType, szMaster) ;
				hmemcpy (ptapinf->szTitle, hpByte + 0xe, 8) ;
				MsxToAnsiBuf (ptapinf->szTitle, 8) ;
				DeleteSpaces (ptapinf->szTitle, 8) ;
				
				return TRUE ;
				}
			
			break ;
		}
	
	ptapinf->szTitle[0] = '\0' ;
	_fstrcpy (ptapinf->szType, szUnknown) ;
	
	return FALSE ;
	}

void SaveRecording (HWND hwnd, BYTE _huge* hpSaveData, DWORD dwSize)
	{
	TAPEINFO		tapinf ;
	
	ptapinf = &tapinf ;
	g_ptphdr = NULL ;
	if (g_szTapeFile[0])
		{
		g_ptphdr = GetTapeInfo (hwnd, g_szTapeFile) ;
		if (g_ptphdr == NULL)
			g_szTapeFile[0] = 0 ;
		}
	GetNameAndType (hpSaveData, &tapinf) ;
	DeActivate () ;
	if (DialogBox (g_hInstance, MAKEINTRESOURCE (IDD_TAPESAVE), hwnd, TapeSaveDlgProc) )
		{
		if (SizeLocal ((void**)&g_ptphdr, (UINT) sizeof (TAPEHEADER) + (g_ptphdr->wTapeRecordings + 1) * sizeof (TAPEINFO) ) )
			{
			if (tapinf.fccChunk = TapeCreateChunk (hwnd, g_szTapeFile, hpSaveData, dwSize) )
				{
				g_ptphdr->tapinf[g_ptphdr->wTapeRecordings++] = tapinf ;
				SetTapeInfo (hwnd, g_szTapeFile, g_ptphdr) ;
				}
			}
		}
	if (g_szTapeFile[0])
		FreeLocal ((void**)&g_ptphdr) ;
	}

// loading from emulator
static void NEAR SetLoadList (HWND hDlg)
	{
	int		i ;
	char		szBuf[256] ;
	
	SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
		LB_RESETCONTENT, 0, 0L) ;
	
	if (g_szTapeFile[0])
		{
		i = 0 ;
		
		while (i < (int)g_ptphdr->wTapeRecordings)
			{
			wsprintf (szBuf, "%s\t%s", 
				(LPSTR)g_ptphdr->tapinf[i].szTitle, 
				(LPSTR)g_ptphdr->tapinf[i].szType) ;
			
			SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
				LB_ADDSTRING, 0, (LPARAM)(LPSTR) szBuf) ;
			i++ ;
			}
		
		SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
			LB_SETCURSEL, 0, 0L) ;
		}
	
	SetWindowText (GetDlgItem (hDlg, IDC_CURTAPEFILE), (g_szTapeFile[0] ?
		(LPCSTR)g_szTapeFile : (LPCSTR)szNoFile) ) ;
	EnableWindow (GetDlgItem (hDlg, IDOK), g_szTapeFile[0] && i) ;
	}

BOOL FAR PASCAL _export TapeLoadDlgProc (HWND hDlg, UINT message, UINT wParam, 
												LONG lParam)
	{
	int		i ;
	char		szBuf[256] ;
	
	switch (message)
		{
		case WM_INITDIALOG :
			i = 125 ;
			SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
				LB_SETTABSTOPS, 1, (LPARAM)(int FAR*) &i) ;
			SetFonts (hDlg) ;
			
			g_ptphdr = NULL ;
			if (g_szTapeFile[0])
				{
				g_ptphdr = GetTapeInfo (hDlg, g_szTapeFile) ;
				if (g_ptphdr == NULL)
					g_szTapeFile[0] = 0 ;
				}
			SetLoadList (hDlg) ;
			
			return TRUE ;
		
		case WM_COMMAND :
			switch (wParam)
				{
				case IDC_RECORDINGSLIST:
					if (HIWORD (lParam) != LBN_DBLCLK)
						break ;
				
				case IDOK:
					i = (int)SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
						LB_GETCURSEL, 0, 0L) ;
					SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
						LB_GETTEXT, (WPARAM)i, (LPARAM)(LPSTR) szBuf) ;
					i = GetListIndex (szBuf, g_ptphdr) ;
					if ((g_hpLoadData = TapeLoadChunk (hDlg, g_szTapeFile, g_ptphdr->tapinf[i].fccChunk, NULL) ) != NULL)
						{
						if (g_szTapeFile[0])
							FreeLocal ((void**)&g_ptphdr) ;
						
						g_bLoading = TRUE ;
					     EndDialog (hDlg, TRUE) ;
					     }
					
					return TRUE ;
				
				case IDCANCEL :
					EndDialog (hDlg, TRUE) ;
					
					if (g_szTapeFile[0])
						FreeLocal ((void**)&g_ptphdr) ;
					
					return TRUE ;
				
				case IDC_FILE:
					if (TapeOpenDlg (hDlg) )
						SetLoadList (hDlg) ;
					
					return TRUE ;
				}
			
			break ;
		}
	
	return FALSE ;
	}

BOOL FAR PASCAL _export TapeTitleDlgProc (HWND hDlg, UINT message, UINT wParam, 
												LONG lParam)
	{
	char		szBuf[256], szBuf2[256] ;
	int		i ;
	
	switch (message)
		{
		case WM_INITDIALOG :
			SetWindowText (GetDlgItem (hDlg, IDC_TITLEEDIT), g_ptphdr->tapinf[nSelect].szTitle) ;
			SetWindowText (GetDlgItem (hDlg, IDC_TYPEEDIT), g_ptphdr->tapinf[nSelect].szType) ;
			assert (g_szTapeFile[0]) ;
			SetWindowText (GetDlgItem (hDlg, IDC_CURTAPEFILE), g_szTapeFile) ;
			SetFonts (hDlg) ;
			
			return TRUE ;
		
		case WM_COMMAND :
			switch (wParam)
				{
				case IDC_TITLEEDIT:
					if (HIWORD (lParam) == EN_CHANGE)
						{
						GetWindowText ((HWND)LOWORD (lParam), szBuf, 256) ;
						if (strlen (szBuf) > 31)
							{
							MessageID (hDlg, IDS_MSGTITLEMAX, MB_OK | MB_ICONEXCLAMATION) ;
							szBuf[31] = '\0' ;
							SetWindowText ((HWND)LOWORD (lParam), szBuf) ;
							}
						}
					
					return TRUE ;
				
				case IDC_TYPEEDIT:
					if (HIWORD (lParam) == EN_CHANGE)
						{
						GetWindowText ((HWND)LOWORD (lParam), szBuf, 256) ;
						if (strlen (szBuf) > 9)
							{
							MessageID (hDlg, IDS_MSGTYPEMAX, MB_OK | MB_ICONEXCLAMATION) ;
							szBuf[9] = '\0' ;
							SetWindowText ((HWND)LOWORD (lParam), szBuf) ;
							}
						}
					
					return TRUE ;
				
				case IDOK :
					GetWindowText (GetDlgItem (hDlg, IDC_TITLEEDIT), szBuf, 256) ;
					GetWindowText (GetDlgItem (hDlg, IDC_TYPEEDIT), szBuf2, 256) ;
					if (!szBuf[0] || !szBuf2[0])
						{
						MessageID (hDlg, IDS_MSGNOTEXT, MB_OK | MB_ICONEXCLAMATION) ;
						return TRUE ;
						}
					
					for (i=0;i<(int)g_ptphdr->wTapeRecordings;i++)
						{
						if (i != nSelect)
							{
							if (!strcmp (szBuf, g_ptphdr->tapinf[i].szTitle) &&
								!strcmp (szBuf2, g_ptphdr->tapinf[i].szType) )
								{
								MessageID (hDlg, IDS_MSGEXIST, MB_OK | MB_ICONEXCLAMATION) ;
								
								return TRUE ;
								}
							}
						}
					
					strcpy (g_ptphdr->tapinf[nSelect].szTitle, szBuf) ;
					strcpy (g_ptphdr->tapinf[nSelect].szType, szBuf2) ;
				case IDCANCEL :
					EndDialog (hDlg, wParam == IDOK) ;
					return TRUE ;
				}
			
			break ;
		}
	
	return FALSE ;
	}

static void NEAR SetButtons (HWND hDlg)
	{
	BOOL		bItems ;
	
	bItems = (SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
		LB_GETCOUNT, 0, 0L) != 0L) ;
	EnableWindow (GetDlgItem (hDlg, IDC_TITLEBTN), bItems) ;
	EnableWindow (GetDlgItem (hDlg, IDC_RECEXPBTN), bItems && waveOutGetNumDevs () ) ;
	EnableWindow (GetDlgItem (hDlg, IDC_RECDELBTN), bItems) ;
	EnableWindow (GetDlgItem (hDlg, IDC_RECIMPBTN), 
		waveInGetNumDevs () && g_szTapeFile[0]) ;
	EnableWindow (GetDlgItem (hDlg, IDC_ADDFILEBTN), g_szTapeFile[0]) ;
	}

BOOL FAR PASCAL _export TapeExportDlgProc (HWND, UINT, UINT, LONG) ;
BOOL FAR PASCAL _export TapeImportDlgProc (HWND, UINT, UINT, LONG) ;
BOOL FAR PASCAL _export TapeAddFileDlgProc (HWND, UINT, UINT, LONG) ;

static void NEAR SetList (HWND hDlg)
	{
	int		i ;
	char		szBuf[256] ;
	
	SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
		LB_RESETCONTENT, 0, 0L) ;
	
	nSelect = 0 ;
	
	if (g_szTapeFile[0])
		{
		i = 0 ;
		
		while (i < (int)g_ptphdr->wTapeRecordings)
			{
			wsprintf (szBuf, "%s\t%s", 
				(LPSTR)g_ptphdr->tapinf[i].szTitle, 
				(LPSTR)g_ptphdr->tapinf[i].szType) ;
			
			SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
				LB_ADDSTRING, 0, (LPARAM)(LPSTR) szBuf) ;
			i++ ;
			}
		
		SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
			LB_SETCURSEL, nSelect, 0L) ;
		}
	
	SetButtons (hDlg) ;
	
	SetWindowText (GetDlgItem (hDlg, IDC_CURTAPEFILE), (g_szTapeFile[0] ?
		(LPCSTR)g_szTapeFile : (LPCSTR)szNoFile) ) ;
	}

BOOL FAR PASCAL _export TapeEditDlgProc (HWND hDlg, UINT message, UINT wParam, 
												LONG lParam)
	{
	int		i, n ;
	char		szBuf[256], szBuf2[256] ;
	
	switch (message)
		{
		case WM_INITDIALOG :
			i = 125 ;
			SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
				LB_SETTABSTOPS, 1, (LPARAM)(int FAR*) &i) ;
			SetFonts (hDlg) ;
			
			g_ptphdr = NULL ;
			if (g_szTapeFile[0])
				{
				g_ptphdr = GetTapeInfo (hDlg, g_szTapeFile) ;
				if (g_ptphdr == NULL)
					g_szTapeFile[0] = 0 ;
				}
			else
				TapeOpenDlg (hDlg) ;
			
			SetList (hDlg) ;
			
			return TRUE ;
		
		case WM_COMMAND :
			switch (wParam)
				{
				case IDC_TITLEBTN:
					i = (int)SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
						LB_GETCURSEL, 0, 0L) ;
					SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
						LB_GETTEXT, i, (LPARAM)(LPSTR) szBuf) ;
					nSelect = GetListIndex (szBuf, g_ptphdr) ;
					
					if (DialogBox (g_hInstance, MAKEINTRESOURCE (IDD_TAPETITLE), hDlg, TapeTitleDlgProc) )
						{
						SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
							LB_DELETESTRING, (WPARAM) i, 0L) ;
						wsprintf (szBuf, "%s\t%s", 
							(LPSTR)g_ptphdr->tapinf[nSelect].szTitle, 
							(LPSTR)g_ptphdr->tapinf[nSelect].szType) ;
						SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
							LB_ADDSTRING, 0, (LPARAM)(LPSTR) szBuf) ;
						SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
							LB_SELECTSTRING, (WPARAM) -1, (LPARAM)(LPSTR) szBuf) ;
						
						SetTapeInfo (hDlg, g_szTapeFile, g_ptphdr) ;
						}
					
					return TRUE ;
				
				case IDC_RECDELBTN:
					i = (int)SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
						LB_GETCURSEL, 0, 0L) ;
					SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
						LB_GETTEXT, i, (LPARAM)(LPSTR) szBuf) ;
					nSelect = GetListIndex (szBuf, g_ptphdr) ;
					
					LoadString (g_hInstance, IDS_MSGRECDEL, szBuf, 256) ;
					wsprintf (szBuf2, szBuf, (LPSTR) g_ptphdr->tapinf[nSelect].szTitle, 
						(LPSTR) g_ptphdr->tapinf[nSelect].szType) ; 
					
					if (IDYES == MessageBox (hDlg, szBuf2, 
						g_szAppName, MB_YESNO | MB_ICONQUESTION | MB_DEFBUTTON2) )
						{
						if (!TapeRemoveChunk (hDlg, g_szTapeFile, g_ptphdr->tapinf[nSelect].fccChunk) )
							break ;
						
						if (!--g_ptphdr->wTapeRecordings)
							SetButtons (hDlg) ;
						
						for (n=nSelect;n<(int)g_ptphdr->wTapeRecordings;n++)
							{
							g_ptphdr->tapinf[n] = g_ptphdr->tapinf[n + 1] ;
							}
						
						SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
							LB_DELETESTRING, i, 0L) ;
						if (i >= (int)SendDlgItemMessage (hDlg, 
							IDC_RECORDINGSLIST, LB_GETCOUNT, 0, 0L) )
							i-- ;
						
						SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
							LB_SETCURSEL, i, 0L) ;
						
						SetTapeInfo (hDlg, g_szTapeFile, g_ptphdr) ;
						
						SetButtons (hDlg) ;
						}
					
					return TRUE ;
				
				case IDC_RECEXPBTN:
					i = (int)SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
						LB_GETCURSEL, 0, 0L) ;
					SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
						LB_GETTEXT, i, (LPARAM)(LPSTR) szBuf) ;
					nSelect = GetListIndex (szBuf, g_ptphdr) ;
					
					hpExp = TapeLoadChunk (hDlg, g_szTapeFile, g_ptphdr->tapinf[nSelect].fccChunk, NULL) ;
					
					if (hpExp != NULL)
						{
						DialogBox (g_hInstance, MAKEINTRESOURCE (IDD_TAPEEXPORT), hDlg, TapeExportDlgProc) ;
		     			FreeGlobal ((void FAR**)&hpExp) ;
						}
					
					return TRUE ;
				
				case IDC_RECIMPBTN:
					DialogBox (g_hInstance, MAKEINTRESOURCE (IDD_TAPEIMPORT), hDlg, TapeImportDlgProc) ;
					
					return TRUE ;
				
				case IDC_ADDFILEBTN:
					DialogBox (g_hInstance, MAKEINTRESOURCE (IDD_TAPEADDFILE), hDlg, TapeAddFileDlgProc) ;
					
					return TRUE ;
				
				case IDCANCEL :
					if (g_szTapeFile[0])
						FreeLocal ((void**)&g_ptphdr) ;
					
					EndDialog (hDlg, TRUE) ;
					
					return TRUE ;
				
				case IDC_FILE:
					if (TapeOpenDlg (hDlg) )
						SetList (hDlg) ;
					
					return TRUE ;
				}
			
			break ;
		}
	
	return FALSE ;
	}

// tape export
BOOL FAR PASCAL _export TapeExportCtrlDlgProc (HWND, UINT, UINT, LONG) ;
static UINT	nDevice ;
static BOOL	b2400bd ;

BOOL FAR PASCAL _export TapeExportDlgProc (HWND hDlg, UINT message, UINT wParam, 
												LONG lParam)
	{
	int			i, n ;
	WAVEOUTCAPS	waveoutcaps ;
	char			szBuf[MAXPNAMELEN] ;
	
	switch (message)
		{
		case WM_INITDIALOG :
			for (i=-1,n=0;i<(int)waveOutGetNumDevs ();i++)
				{
				if (!waveOutGetDevCaps (i, &waveoutcaps, sizeof (WAVEOUTCAPS) ) )
					{
					SendDlgItemMessage (hDlg, IDC_WAVEOUTPUTCMB, CB_ADDSTRING, 
						0, (LPARAM) (LPSTR) waveoutcaps.szPname) ;
					n = 1 ;
					}
				}
			if (n == 0)
				{
				MessageID (hDlg, IDS_MSGNOWAVEOUT, MB_ICONEXCLAMATION) ;
				EndDialog (hDlg, FALSE) ;
				}
			
			SendDlgItemMessage (hDlg, IDC_WAVEOUTPUTCMB, CB_SETCURSEL, 0, 0L) ;
			SendDlgItemMessage (hDlg, IDC_2400BDRAD, BM_SETCHECK, 1, 0L) ;
			PostMessage (hDlg, WM_COMMAND, IDC_WAVEOUTPUTCMB, ((LPARAM) CBN_SELCHANGE) << 16) ;
			
			return TRUE ;
		
		case WM_COMMAND:
			switch (wParam)
				{
				case IDC_WAVEOUTPUTCMB:
					if (HIWORD (lParam) == CBN_SELCHANGE)
						{
						SendDlgItemMessage (hDlg, IDC_WAVEOUTPUTCMB, CB_GETLBTEXT, 
							(WPARAM) SendDlgItemMessage (hDlg, IDC_WAVEOUTPUTCMB, CB_GETCURSEL, 
							0, 0L), (LPARAM)(LPCSTR) szBuf) ;
						for (i=-1;i<(int)waveOutGetNumDevs ();i++)
							{
							if (!waveOutGetDevCaps (i, &waveoutcaps, sizeof (WAVEOUTCAPS) ) && 
								!strcmp (waveoutcaps.szPname, szBuf) )
								{
								nDevice = i ;
								break ;
								}
							}
						
						n = FALSE ;
						if (! (waveoutcaps.dwFormats & WAVE_FORMAT_2M08) )
							{
							SendDlgItemMessage (hDlg, IDC_1200BDRAD, BM_SETCHECK, 0, 0L) ;
							EnableWindow (GetDlgItem (hDlg, IDC_1200BDRAD), FALSE) ;
							SendDlgItemMessage (hDlg, IDC_2400BDRAD, BM_SETCHECK, 1, 0L) ;
							}
						else
							{
							EnableWindow (GetDlgItem (hDlg, IDC_1200BDRAD), TRUE) ;
							n = TRUE ;
							}
						
						if (! (waveoutcaps.dwFormats & WAVE_FORMAT_4M08) )
							{
							SendDlgItemMessage (hDlg, IDC_2400BDRAD, BM_SETCHECK, 0, 0L) ;
							EnableWindow (GetDlgItem (hDlg, IDC_2400BDRAD), FALSE) ;
							SendDlgItemMessage (hDlg, IDC_1200BDRAD, BM_SETCHECK, 1, 0L) ;
							}
						else
							{
							EnableWindow (GetDlgItem (hDlg, IDC_2400BDRAD), TRUE) ;
							n = TRUE ;
							}
						
						EnableWindow (GetDlgItem (hDlg, IDOK), n) ;
						
						return TRUE ;
						}
					
					break ;
				
				case IDOK:
					waveOutGetDevCaps (nDevice, &waveoutcaps, sizeof (WAVEOUTCAPS) ) ;
					b2400bd = (SendDlgItemMessage (hDlg, IDC_2400BDRAD, BM_GETCHECK, 0, 0L) != 0L) ;
					if (waveoutcaps.dwSupport & WAVECAPS_SYNC)
						MessageID (hDlg, IDS_MSGNOSYNC, MB_ICONEXCLAMATION) ;
					else	if (MessageID (hDlg, IDS_TAPEEXPORT, MB_OKCANCEL | MB_ICONINFORMATION) == IDOK)
						{
						EndDialog (hDlg, TRUE) ;
						
						DialogBox (g_hInstance, MAKEINTRESOURCE (IDD_TAPEEXPORTCTRL), GetParent (hDlg), TapeExportCtrlDlgProc) ;
						}
					
					return TRUE ;
				
				case IDCANCEL:
					EndDialog (hDlg, FALSE) ;
					
					return TRUE ;
				}
			
			break ;
		}
	
	return FALSE ;
	}

// tape waveform data
const static BYTE BASED_CODE byLO[] = { 96, 64, 32, 0, 0, 0, 32, 64, 96, 
	160, 191, 223, 255, 255, 255, 223, 191, 160 } ;
const static BYTE BASED_CODE byHI[] = { 64, 0, 0, 64, 128, 191, 255, 255, 191, 
	64, 0, 0, 64, 128, 191, 255, 255, 191 } ;
static HWAVEOUT	hWaveOut ;
static DWORD		dwTotal, dwVolume ;

static void NEAR WriteBlock (HWND hDlg, BOOL bFirst, LPWAVEHDR lpWaveHdr)
	{
	static DWORD	dwLoad ;
	int			i, nFreq ;
	static int	nHeader, nBlank ;
	static BOOL	bEnd ;
	BYTE			byData ;
	LPBYTE		lpData ;
	
	assert (sizeof (byLO) == sizeof (byHI) ) ;
	
	if (bFirst)
		{
		// first block
		hpExpPtr = hpExp ;
		dwTotal = 8000 ;
		while (1)
			{
			dwLoad = *(DWORD _huge*)hpExpPtr ;
			if (dwLoad == -1L)
				break ;
			
			hpExpPtr += dwLoad + 4L ;
			dwTotal += 8000 + dwLoad * 11 ;
			}
		
		dwTotal *= sizeof (byLO) ;
		
		hpExpPtr = hpExp ;
		dwLoad = *(DWORD _huge*)hpExpPtr ;
		hpExpPtr += 4 ;
		bEnd = (dwLoad == -1L) ;
		nBlank = 0 ;
		nHeader = 16000 ;
		for (i=0;i<8;i++)
			WriteBlock (hDlg, FALSE, NULL) ;
		}
	
	if (bEnd)
		return ;
	
	if (lpWaveHdr == NULL)
		{
		if (!NewGlobal ((void FAR**)&lpWaveHdr, GMEM_ZEROINIT | GMEM_MOVEABLE | GMEM_SHARE, sizeof (WAVEHDR) ) )
			return ;
		if (!NewGlobal ((void FAR**)&lpData, GMEM_MOVEABLE | GMEM_SHARE, 0xffff) )
			{
			FreeGlobal ((void FAR**)&lpWaveHdr) ;
			return ;
			}
		lpWaveHdr->lpData = lpData ;
		}
	else
		lpData = lpWaveHdr->lpData ;
	
	nFreq = 0xffff / sizeof (byLO) ;
	
header:
	while (nBlank && nFreq)
		{
		_fmemset (lpData, 128, sizeof (byLO)) ;
		
		lpData += sizeof (byLO) ;
		nBlank-- ;
		nFreq-- ;
		}
	
	while (nHeader && nFreq)
		{
		_fmemcpy (lpData, byHI, sizeof (byLO)) ;
		
		lpData += sizeof (byLO) ;
		nHeader-- ;
		nFreq--;
		}
	
	while (nFreq > 11)
		{
		if (dwLoad--)
			{
			_fmemcpy (lpData, byLO, sizeof (byLO)) ;
			lpData += sizeof (byLO) ;
			byData = *hpExpPtr++ ;
			for (i=0;i<8;i++)
				{
				_fmemcpy (lpData, ((byData & 1) ? byHI : byLO), sizeof (byLO)) ;
				lpData += sizeof (byLO) ;
				byData >>= 1 ;
				}
			_fmemcpy (lpData, byHI, sizeof (byLO)) ;
			lpData += sizeof (byLO) ;
			_fmemcpy (lpData, byHI, sizeof (byLO)) ;
			lpData += sizeof (byLO) ;
			
			nFreq -= 11 ;
			}
		else
			{
			dwLoad = *(DWORD _huge*)hpExpPtr ;
			hpExpPtr += 4L ;
			
			nBlank = 4000 ;
			nHeader = 4000 ;
			if (dwLoad != -1L)
				goto header ;
			else
				{
				bEnd = TRUE ;
				break ;
				}
			}
		}
	
	lpWaveHdr->dwBufferLength = (DWORD) LOWORD ((DWORD)(lpData)) ;
	waveOutPrepareHeader (hWaveOut, lpWaveHdr, sizeof (WAVEHDR) ) ;
	waveOutWrite (hWaveOut, lpWaveHdr, sizeof (WAVEHDR) ) ;
	}

static void NEAR StopPlay (HWND hDlg)
	{
	waveOutReset (hWaveOut) ;
	waveOutClose (hWaveOut) ;
	waveOutSetVolume (nDevice, dwVolume) ;
		     			
	SetWindowText (GetDlgItem (hDlg, IDC_START), "Start") ;
	}

BOOL FAR PASCAL _export TapeExportCtrlDlgProc (HWND hDlg, UINT message, UINT wParam, 
												LONG lParam)
	{
     PAINTSTRUCT		ps ;
   	PCMWAVEFORMAT		pcmwave ;
   	MMTIME			mmtime ;
   	int				i, x, y ;
   	WORD				n ;
   	DWORD			dw ;
   	RECT				rc, rc2 ;
   	static BOOL		bPlay  ;
   	static int		nPerDone ;
	char				szBuf[MAXERRORLENGTH] ;
   	LPBYTE			lpData ;
     
     switch (message)
     	{
     	case WM_INITDIALOG:
			nPerDone = 0 ;
			bPlay = FALSE ;
			if (!SetTimer (hDlg, 1, 500, NULL) )
				MessageID (hDlg, IDS_MSGTAPETIMER, MB_OK | MB_ICONEXCLAMATION) ;
     		
     		return TRUE ;
     	
     	case WM_PAINT:
     		BeginPaint (hDlg, &ps) ;
     		
     		GetWindowRect (GetDlgItem (hDlg, IDC_PERCENTDONE), &rc) ;
			ScreenToClient (hDlg, (LPPOINT)&rc.left) ;
			ScreenToClient (hDlg, (LPPOINT)&rc.right) ;
			
     		wsprintf (szBuf, "%d %%", nPerDone) ;
     		dw = GetTextExtent (ps.hdc, szBuf, strlen (szBuf) ) ;
     		x = rc.left + (rc.right - rc.left - LOWORD (dw) ) / 2 ;
     		y = rc.top + (rc.bottom - rc.top - HIWORD (dw) ) / 2 ;
     		
     		rc2 = rc ;
     		rc2.right = rc.left + ( (rc.right - rc.left) * nPerDone / 100) ;
     		SetBkColor (ps.hdc, GetSysColor (COLOR_ACTIVECAPTION) ) ;
     		SetTextColor (ps.hdc, GetSysColor (COLOR_BTNFACE) ) ;
     		ExtTextOut (ps.hdc, x, y, ETO_CLIPPED | ETO_OPAQUE, 
     			&rc2, szBuf, strlen (szBuf), NULL) ;
     		
     		rc2.left = rc2.right + 1 ;
     		rc2.right = rc.right ;
     		SetBkColor (ps.hdc, GetSysColor (COLOR_BTNFACE) ) ;
     		SetTextColor (ps.hdc, GetSysColor (COLOR_ACTIVECAPTION) ) ;
     		ExtTextOut (ps.hdc, x, y, ETO_CLIPPED | ETO_OPAQUE, 
     			&rc2, szBuf, strlen (szBuf), NULL) ;
     		
     		EndPaint (hDlg, &ps) ;
     		
     		return TRUE ;
     	
     	case WM_COMMAND:
     		switch (wParam)
     			{
     			case IDC_START:
		     		if (!bPlay)
			     		{
	    					pcmwave.wf.wFormatTag  = WAVE_FORMAT_PCM ;
		     			pcmwave.wf.nChannels   = 1 ;
			     		pcmwave.wf.nBlockAlign = 1 ;
     					pcmwave.wBitsPerSample = 8 ;
     			     	
		     			if (b2400bd)
		     				{
			     			pcmwave.wf.nSamplesPerSec  = 44100 ;
		     				pcmwave.wf.nAvgBytesPerSec = 44100 ;
		     				}
		     			else
		     				{
			     			pcmwave.wf.nSamplesPerSec  = 22050 ;
		     				pcmwave.wf.nAvgBytesPerSec = 22050 ;
		     				}
     					
     					n = waveOutOpen (&hWaveOut, nDevice, (LPWAVEFORMAT)&pcmwave, 
     						(DWORD)(WORD)hDlg, 0L, CALLBACK_WINDOW) ;
     					
     					if (n)
     						{
     						waveOutGetErrorText (n, szBuf, sizeof (szBuf) ) ;
     						MessageBox (hDlg, szBuf, g_szAppName, MB_OK | MB_ICONEXCLAMATION) ;
     						}
	     				else
     						{
     						// store current volume and set to max
     						waveOutGetVolume (nDevice, &dwVolume) ;
     						waveOutSetVolume (nDevice, 0xffffffff) ;
     						SetWindowText (GetDlgItem (hDlg, IDC_START), "Stop") ;
     						bPlay = TRUE ;
     						}
     					}
     				else
     					{
    						bPlay = FALSE ;
    						
    						StopPlay (hDlg) ;
    						}
     				
     				return TRUE ;
     			
     			case IDCANCEL:
     				if (bPlay)
     					{
     					if (IDYES == MessageID (hDlg, IDS_MSGPLAYING, MB_YESNO | MB_ICONQUESTION | MB_DEFBUTTON2) )
     						{
	     					bPlay = FALSE ;
     						
	    						StopPlay (hDlg) ;
	     					
	     					EndDialog (hDlg, TRUE) ;
	     					}
     					}
    					else
    						EndDialog (hDlg, TRUE) ;
     				
     				return TRUE ;
     			}
     		
     		break ;
     	
     	case WM_TIMER:
			if (bPlay)
				{
				mmtime.wType = TIME_SAMPLES ;
				waveOutGetPosition (hWaveOut, &mmtime, sizeof (MMTIME) ) ;
				
				i = LOWORD (mmtime.u.sample * 100L / dwTotal) ;
				
				if (mmtime.u.sample == dwTotal)
					{
					bPlay = FALSE ;
     				
					StopPlay (hDlg) ;
					}
				}
			else
				i = 0 ;
			
			if (i != nPerDone)
				{
				nPerDone = i ;
				
				GetWindowRect (GetDlgItem (hDlg, IDC_PERCENTDONE), &rc) ;
				ScreenToClient (hDlg, (LPPOINT)&rc.left) ;
				ScreenToClient (hDlg, (LPPOINT)&rc.right) ;
				InvalidateRect (hDlg, &rc, FALSE) ;
				}
     		
     		return TRUE ;
     	
     	case MM_WOM_OPEN:
     		WriteBlock (hDlg, TRUE, NULL) ;
     		
     		return TRUE ;
     	
     	case MM_WOM_DONE:
			waveOutUnprepareHeader ((HWAVEOUT) wParam,
				(LPWAVEHDR) lParam, sizeof (WAVEHDR) ) ;
			
     		if (bPlay)
     			WriteBlock (hDlg, FALSE, (LPWAVEHDR) lParam) ;
			else
				{
				lpData = ((LPWAVEHDR)lParam)->lpData ;
				FreeGlobal ((void FAR**)&lpData) ;
				FreeGlobal ((void FAR**)&lParam) ;
				}
			
			return TRUE ;
		
		case WM_CLOSE:
			if (bPlay)
				{
				if (IDYES == MessageID (hDlg, IDS_MSGPLAYING, MB_YESNO | MB_ICONQUESTION | MB_DEFBUTTON2) )
					{
					bPlay = FALSE ;
   					
   					StopPlay (hDlg) ;
					
					EndDialog (hDlg, TRUE) ;
					}
				}
			else
				EndDialog (hDlg, TRUE) ;
			
			return TRUE ;
		
		case WM_DESTROY:
			KillTimer (hDlg, 1) ;
			
			return TRUE ;
		}
	
	return FALSE ;
	}

// tape import
static BOOL		bFirst ;
static BYTE _huge*	hpbyData, _huge* hpbyDataPtr ;
static DWORD		dwBlock ;
static int		nOldState ;

static void NEAR CheckSize (HWND hDlg)
	{
	DWORD	dwOffset = (hpbyDataPtr - hpbyData) ;
	DWORD	dwSize   = dwOffset + 8 + dwBlock ;
	
	if (GlobalSize (GlobalPtrHandle (hpbyData) ) < dwSize)
		{
		if (!SizeGlobal ((void FAR**)&hpbyData, dwSize + 0x4000) )
			{
			SendMessage (hDlg, WM_COMMAND, IDC_START, 0L) ;
			return ;
			}
		
		hpbyDataPtr = hpbyData + dwOffset ;
		}
	}

static void NEAR EndBlock (HWND hDlg)
	{
	TAPEINFO		tapinf ;
	
	if (dwBlock)
		{
		CheckSize (hDlg) ;
			
		*(DWORD _huge*)hpbyDataPtr = dwBlock ;
		
		if (GetNameAndType (hpbyDataPtr, &tapinf) )
			{
			SetWindowText (GetDlgItem (hDlg, IDC_FILEFOUNDTITLE), tapinf.szTitle) ;
			SetWindowText (GetDlgItem (hDlg, IDC_FILEFOUNDTYPE), tapinf.szType) ;
			}
		
		hpbyDataPtr += dwBlock + 4 ;
		dwBlock = 0L ;
		}
	}

static _inline BOOL NEAR IsNear (int base, int n)
	{
	// may be 30% off
	return (n * 10 >= base * 7 && n * 10 <=  base * 13) ;
	}

static void NEAR SetStateText (HWND hDlg, int nState)
	{
	static char	*szState[] = {
		"Waiting for Header", "Loading Header", 
		"Loading Data", "Load Error" } ;
	
	if (nState != nOldState && !(nState == 0 && nOldState == 3) )
		{
		nOldState = nState ;
		
		SetWindowText (GetDlgItem (hDlg, IDC_IMPORTMESSAGE), szState[nState]) ;
		}
	}

static void NEAR ReadBlock (HWND hDlg, LPBYTE lpbyBlock, WORD wSize)
	{
	static int	nSubState, nHeaderBlocks, nBit, nTransitions, nCount ;
	static WORD	wBlockSize, wCurBlockSize, wLastBlockSize ;
	static BOOL	bFase ;
	static BYTE	byData ;
	static long	lSize ;
	
	if (bFirst)
		{
		bFirst = FALSE ;
		
		nHeaderBlocks = nSubState = 0 ;
		nOldState = -1 ;
		SetStateText (hDlg, 0) ;
		if (!NewGlobal ((void FAR**)&hpbyData, GHND, 0x4000) )
			{
			SendMessage (hDlg, WM_COMMAND, IDC_START, 0L) ;
			return ;
			}
		
		hpbyDataPtr = hpbyData ;
		dwBlock = 0L ;
		
		goto NextHeaderBlock ;
		}
	
	switch (nSubState)
		{
NextHeaderBlock:
			wCurBlockSize = 0 ;
		case 0:
			nSubState = 0 ;
			while ( (*lpbyBlock > 128) == bFase)
				{
				wCurBlockSize++ ;
				lpbyBlock++ ;
				if (!wSize--)
					return ;
				}
			
			bFase = !bFase ;
			
			if (wCurBlockSize > 40)
				{
				nHeaderBlocks = 0 ;
				
				goto NextHeaderBlock ;
				}
			
			if (IsNear (wBlockSize, wCurBlockSize) )
				{
				wBlockSize = wCurBlockSize ;
				if (nHeaderBlocks++ < 1111)
					goto NextHeaderBlock ;
				}
			else 
				{
				nHeaderBlocks = 0 ;
				wBlockSize = wCurBlockSize ;
				
				goto NextHeaderBlock ;
				}
			
			SetStateText (hDlg, 1) ;
			lSize = 0L ;
			nCount = 0 ;
NextHeader:
			wCurBlockSize = 0 ;
		case 5:
			nSubState = 5 ;
			while ( (*lpbyBlock > 128) == bFase)
				{
				wCurBlockSize++ ;
				lpbyBlock++ ;
				if (!wSize--)
					return ;
				}
			
			bFase = !bFase ;
			
			if (wCurBlockSize > 40)
				{
				nHeaderBlocks = 0 ;
				SetStateText (hDlg, 0) ;
				
				goto NextHeaderBlock ;
			     }
			
			lSize += (long)wCurBlockSize ;
			if (++nCount < 256)
				goto NextHeader ;
		     
		     wBlockSize = (WORD) (lSize / 256) ;
NextByteHeader:
			wCurBlockSize = 0 ;
		case 1:
			nSubState = 1 ;
			while ( (*lpbyBlock > 128) == bFase)
				{
				wCurBlockSize++ ;
				lpbyBlock++ ;
				if (!wSize--)
					return ;
				}
			
			bFase = !bFase ;
			
			if (wCurBlockSize > 40 || wCurBlockSize * 10 < wBlockSize * 8)
				{
				EndBlock (hDlg) ;
				
				nHeaderBlocks = 0 ;
				SetStateText (hDlg, 0) ;
				
				goto NextHeaderBlock ;
			     }
			
			if (10 * (wCurBlockSize + wLastBlockSize) < 38 * wBlockSize)
				{
				wLastBlockSize = wCurBlockSize ;
				
				goto NextByteHeader ;
				}
			
			nBit = 0 ;
NextBit:
			wCurBlockSize = 3 * wBlockSize ;
		case 3:
			nSubState = 3 ;
			nTransitions = 0 ;
NextTrans:
			while ( (*lpbyBlock > 128) == bFase)
				{
				if (!--wCurBlockSize)
					break ;
				lpbyBlock++ ;
				if (!wSize--)
					return ;
				}
			
			if (wCurBlockSize)
				{
				bFase = !bFase ;
				
				nTransitions++ ;
				goto NextTrans ;
				}
			
			if (nTransitions >= 4)
				{
				EndBlock (hDlg) ;
				SetStateText (hDlg, 3) ;
				nHeaderBlocks = 0 ;
				
				goto NextHeaderBlock ;
			     }
			
			byData /= 2 ;
			byData += (BYTE) ( (nTransitions > 1) * 128) ;
			
			// pass the remaining transitions
			nTransitions = ( (nTransitions > 1) ? 4 : 2) - nTransitions ;
NextPass:		
			wCurBlockSize = 0 ;
		case 4:
			nSubState = 4 ;
			while ( (*lpbyBlock > 128) == bFase)
				{
				wCurBlockSize++ ;
				lpbyBlock++ ;
				if (!wSize--)
					return ;
				}
			
			bFase = !bFase ;
			
			if (wCurBlockSize > 40)
				{
				EndBlock (hDlg) ;
				
				nHeaderBlocks = 0 ;
				SetStateText (hDlg, 0) ;
				
				goto NextHeaderBlock ;
				}
			
			if (--nTransitions)
				goto NextPass ;			
			
			SetStateText (hDlg, 2) ;
			
			if (++nBit < 8)
				goto NextBit ;
			
			// add byte to recording
			CheckSize (hDlg) ;
			hpbyDataPtr[4+dwBlock++] = byData ;
			
			goto NextByteHeader ;
		}
	}

static void NEAR SaveImport (HWND, BYTE _huge*, DWORD) ;

static BOOL	bGetROM ;

BOOL FAR PASCAL _export TapeImportDlgProc (HWND hDlg, UINT message, UINT wParam, 
												LONG lParam)
	{
	int			i ;
	WORD			n ;
   	char			szBuf[max (MAXERRORLENGTH, MAXPNAMELEN)] ;
	WAVEINCAPS	wavincaps ;
	static BOOL	bRec, bAsk ;
	static HWAVEIN	hWaveIn ;
	PCMWAVEFORMAT	pcmwave ;
	LPWAVEHDR		lpWaveHdr ;
	static int	nBlock ;
   	LPBYTE		lpData ;
	
	switch (message)
		{
		case WM_INITDIALOG:
			for (n=0,i=-1;i<(int)waveInGetNumDevs ();i++)
				{
				if (!waveInGetDevCaps (i, &wavincaps, sizeof (WAVEINCAPS) ) && 
					(wavincaps.dwFormats & (WAVE_FORMAT_2M08 | WAVE_FORMAT_4M08) ) )
					{
					SendDlgItemMessage (hDlg, IDC_WAVEINPUTCMB, CB_ADDSTRING, 
						0, (LPARAM) (LPSTR) wavincaps.szPname) ;
					n = 1 ;
					}
				}
			
			if (n == 0)
				{
				MessageID (hDlg, IDS_MSGNOWAVEIN, MB_ICONEXCLAMATION) ;
				EndDialog (hDlg, FALSE) ;
				}
			
			SendDlgItemMessage (hDlg, IDC_WAVEINPUTCMB, CB_SETCURSEL, 0, 0L) ;
			bRec = bAsk = FALSE ;
			
			return TRUE ;
		
		case WM_COMMAND:
			switch (wParam)
				{
				case IDC_START:
					if (!bRec)
						{
						if (!bAsk && IDCANCEL == MessageID (hDlg, 
							IDS_TAPEIMPORT, MB_OKCANCEL | MB_ICONINFORMATION) )
								return TRUE ;
						bAsk = TRUE ;
						
						SendDlgItemMessage (hDlg, IDC_WAVEINPUTCMB, CB_GETLBTEXT, 
							(WPARAM) SendDlgItemMessage (hDlg, IDC_WAVEINPUTCMB, CB_GETCURSEL, 
							0, 0L), (LPARAM)(LPCSTR) szBuf) ;
						for (i=-1;i<(int)waveInGetNumDevs ();i++)
							{
							if (	!waveInGetDevCaps (i, &wavincaps, sizeof (WAVEINCAPS) ) && 
								!strcmp (wavincaps.szPname, szBuf) )
								break ;
							}
						
						pcmwave.wf.wFormatTag  = WAVE_FORMAT_PCM ;
						pcmwave.wf.nChannels   = 1 ;
						pcmwave.wf.nBlockAlign = 1 ;
						pcmwave.wBitsPerSample = 8 ;
						
						if (wavincaps.dwFormats & WAVE_FORMAT_4M08)
							{
			     			pcmwave.wf.nSamplesPerSec  = (DWORD) 44100 ;
		     				pcmwave.wf.nAvgBytesPerSec = (DWORD) 44100 ;
							}
						else
							{
			     			assert (wavincaps.dwFormats & WAVE_FORMAT_2M08) ;
			     			
			     			pcmwave.wf.nSamplesPerSec  = (DWORD) 22050 ;
		     				pcmwave.wf.nAvgBytesPerSec = (DWORD) 22050 ;
							}
						
     					n = waveInOpen (&hWaveIn, i, (LPWAVEFORMAT)&pcmwave, 
     						(DWORD)(WORD)hDlg, 0L, CALLBACK_WINDOW) ;
     					
     					if (n)
     						{
     						waveInGetErrorText (n, szBuf, sizeof (szBuf) ) ;
     						MessageBox (hDlg, szBuf, g_szAppName, MB_OK | MB_ICONEXCLAMATION) ;
     						}
	     				else
     						{
     						SetWindowText (GetDlgItem (hDlg, IDC_START), "Stop") ;
							if (GetFocus () == GetDlgItem (hDlg, IDC_WAVEINPUTCMB) )
								SetFocus (GetDlgItem (hDlg, IDC_START) ) ;
							EnableWindow (GetDlgItem (hDlg, IDC_WAVEINPUTCMB), FALSE) ;
							bFirst = bRec = TRUE ;
     						}
						}
					else
						{
						waveInReset (hWaveIn) ;
						waveInClose (hWaveIn) ;
						EnableWindow (GetDlgItem (hDlg, IDC_WAVEINPUTCMB), TRUE) ;
						
						SetWindowText (GetDlgItem (hDlg, IDC_START), "Start") ;
						SetWindowText (GetDlgItem (hDlg, IDC_IMPORTMESSAGE), "") ;
						SetWindowText (GetDlgItem (hDlg, IDC_FILEFOUNDTITLE), "") ;
						SetWindowText (GetDlgItem (hDlg, IDC_FILEFOUNDTYPE), "") ;
						
						SaveImport (hDlg, hpbyData, (DWORD)(hpbyDataPtr - hpbyData) + 4) ;
						
						FreeGlobal ((void FAR**)&hpbyData) ;
						
						bRec = FALSE ;
						}
					
					return TRUE ;
				
				case IDCANCEL:
					if (bRec)
						{
						if (IDYES == MessageID (hDlg, IDS_MSGRECING, MB_YESNO | MB_ICONQUESTION | MB_DEFBUTTON2) )
							{
							waveInReset (hWaveIn) ;
							waveInClose (hWaveIn) ;
							EnableWindow (GetDlgItem (hDlg, IDC_WAVEINPUTCMB), TRUE) ;
							
							SetWindowText (GetDlgItem (hDlg, IDC_START), "Start") ;
							SetWindowText (GetDlgItem (hDlg, IDC_IMPORTMESSAGE), "") ;
							SetWindowText (GetDlgItem (hDlg, IDC_FILEFOUNDTITLE), "") ;
							SetWindowText (GetDlgItem (hDlg, IDC_FILEFOUNDTYPE), "") ;
							
							SaveImport (hDlg, hpbyData, (DWORD)(hpbyDataPtr - hpbyData) + 4) ;
							
							FreeGlobal ((void FAR**)&hpbyData) ;
							
							bRec = FALSE ;
							
							EndDialog (hDlg, TRUE) ;
							}
						}
					else
						EndDialog (hDlg, TRUE) ;
					
					return TRUE ;
				}
			
			break ;
		
		case MM_WIM_OPEN:
			for (nBlock=0;nBlock<4;nBlock++)
				{
				if (!NewGlobal ((void FAR**)&lpWaveHdr, GMEM_SHARE | GMEM_MOVEABLE | GMEM_ZEROINIT, sizeof (WAVEHDR) ) )
					break ;
				if (!NewGlobal ((void FAR**)&lpData, GMEM_SHARE | GMEM_MOVEABLE, 0x3fffL) )
					{
					FreeGlobal ((void FAR**)&lpWaveHdr) ;
					break ;
					}
				lpWaveHdr->lpData = lpData ;
				lpWaveHdr->dwBufferLength = 0x3fffL ;
			     waveInPrepareHeader (hWaveIn, lpWaveHdr, sizeof (WAVEHDR) ) ;
			     waveInAddBuffer (hWaveIn, lpWaveHdr, sizeof (WAVEHDR) ) ;
				}
			
     		waveInStart (hWaveIn) ;
			
			return TRUE ;
		
		case MM_WIM_DATA:
			lpWaveHdr = (LPWAVEHDR) lParam ;
			
			n = LOWORD (lpWaveHdr->dwBytesRecorded) ;
			ReadBlock (hDlg, lpWaveHdr->lpData, n ? n - 1 : 0) ;
			
			// done with it
			if (bRec)
				{
				// return it
				waveInAddBuffer (hWaveIn, lpWaveHdr, sizeof (WAVEHDR) ) ;
				}
			else
				{
				// free it
				waveInUnprepareHeader (hWaveIn, lpWaveHdr, sizeof (WAVEHDR) ) ;
				lpData = lpWaveHdr->lpData ;
				FreeGlobal ((void FAR**)&lpData) ;
				FreeGlobal ((void FAR**)&lpWaveHdr) ;
				}
			
			return TRUE ;
		}
	
	return FALSE ;
	}

static BOOL NEAR GetROMname (BYTE _huge* hpByte, char *szFile)
	{
	BYTE		byte ;
	int		i ;
	
	switch (*(DWORD _huge*)hpByte)
		{
		case 0x18L:
			// ROM saving from GetROM ?
			byte = hpByte[4] ;
			for (i=5;i<0x14;i++)
				if (hpByte[i] != byte)
					{
					byte = 0 ;
					break ;
					}
			
			if (byte == 0x55)
				{
				hmemcpy (szFile, hpByte + 0x14, 8) ;
				MsxToAnsiBuf (szFile, 8) ;
				for (i=0;i<8;i++)
					{
					if (szFile[i] <= ' ')
						break ;
					}
				
				strcpy (&szFile[i], ".ROM") ;
				
				return TRUE ;
				}
			
			break ;
		}
	
	return FALSE ;
	}

static void NEAR SaveROM (HWND hDlg, char *szFile, BYTE _huge* hpbyData, DWORD dwSize)
	{
	DWORD	dw ;
	long		lSize ;
	HFILE	hFile ;
	
	// skip filename header
	dw = *(DWORD _huge*)hpbyData ;
	hpbyData += dw + 4 ;
	
	// first byte is size / 8Kb
	lSize = (long)hpbyData[4] * 0x2000L ;
	dw = *(DWORD _huge*)hpbyData ;
	
	hFile = _lcreat (szFile, 0) ;
	if (HFILE_ERROR == hFile)
		goto err1 ;
	
	if ( (long)(dw - 1) != _hwrite (hFile, hpbyData + 5, dw - 1) )
		goto err2 ;
	
	hpbyData += dw + 4 ;
	lSize -= (dw - 1) ;
	
	while (lSize > 0)
		{
		dw = *(DWORD _huge*)hpbyData ;
		if (!dw)
			break ;
		
		if ( (long)dw != _hwrite (hFile, hpbyData + 4, dw) )
			goto err2 ;
		
		hpbyData += dw + 4 ;
		lSize -= dw ;
		}
	
	_lclose (hFile) ;
	
	if (lSize)
		{
		OpenFile (szFile, NULL, OF_DELETE) ;
		MessageID (hDlg, IDS_DATACORRUPT, MB_ICONEXCLAMATION | MB_OK) ;
		}
	
	return ;
	
err2:
	_lclose (hFile) ;
	OpenFile (szFile, NULL, OF_DELETE) ;
err1:
	MessageID (hDlg, IDS_SAVEFAIL, MB_ICONEXCLAMATION | MB_OK) ;
	}

static void NEAR SaveImport (HWND hDlg, BYTE _huge* hpbyData, DWORD dwImpSize)
	{
	DWORD		dw, dwSize, dwBlock ;
	TAPEINFO		tapinf ;
	BYTE _huge*	hpbyTemp ;
	char			szFile[_MAX_PATH] ;
	OPENFILENAME	ofn ;
	static const char BASED_CODE
				szFilter[] = "ROM Files (*.rom)\0*.rom\0All Files (*.*)\0*.*\0" ;
	
	if (bGetROM)
		{
		dwSize = dw = 0 ;
		while (1)
			{
			dwBlock = *(DWORD _huge*)&hpbyData[dw + dwSize] ;
			if (dwSize)
				{
				if (!dwBlock || GetROMname (&hpbyData[dw + dwSize], szFile) )
					{
					memset (&ofn, 0, sizeof (OPENFILENAME) ) ;
					ofn.lStructSize = sizeof (OPENFILENAME) ;
					ofn.hwndOwner = hDlg ;
					ofn.lpstrFilter = szFilter ;
					ofn.lpstrFile = szFile ;
					ofn.nMaxFile = sizeof (szFile) ;
					ofn.Flags = OFN_PATHMUSTEXIST | OFN_HIDEREADONLY | 
						OFN_OVERWRITEPROMPT ;
					
					GetROMname (&hpbyData[dw], szFile) ;
					DeActivate () ;
					if (GetSaveFileName (&ofn) )
						SaveROM (hDlg, szFile, hpbyData + dw, dwSize) ;
					}
				}
			
			// infinite loop, check for end of block!
			dwSize += dwBlock + 4L ;
			if (!dwBlock)
				break ;
			}
		}
	else	if (*((DWORD _huge*)hpbyData) )
		{
		dwSize = dw = 0 ;
		while (1)
			{
			dwBlock = *(DWORD _huge*)&hpbyData[dw + dwSize] ;
			if (dwSize && (!dwBlock || GetNameAndType (&hpbyData[dw + dwSize], &tapinf) ) )
				{
				DeActivate () ;
				ptapinf = &tapinf ;
				GetNameAndType (&hpbyData[dw], &tapinf) ;
				if (DialogBox (g_hInstance, MAKEINTRESOURCE (IDD_TAPESAVE), hDlg, TapeSaveDlgProc) )
					{
					// just make sure block ends an 0
					if (NewGlobal ((void FAR**)&hpbyTemp, GMEM_MOVEABLE, dwSize + 4L) )
						{
						hmemcpy (hpbyTemp, hpbyData + dw, dwSize) ;
						*(DWORD _huge*)(hpbyTemp + dwSize) = (DWORD)-1L ;
						if (SizeLocal ((void**)&g_ptphdr, (UINT) sizeof (TAPEHEADER) + 
							(g_ptphdr->wTapeRecordings + 1) * sizeof (TAPEINFO) ) )
							{
							if (tapinf.fccChunk = 
								TapeCreateChunk (hDlg, g_szTapeFile, hpbyTemp, dwSize + 4L) )
								{
								g_ptphdr->tapinf[g_ptphdr->wTapeRecordings++] = tapinf ;
								SetTapeInfo (hDlg, g_szTapeFile, g_ptphdr) ;
								}
							FreeGlobal ((void FAR**)&hpbyTemp) ;
							}
						}
					}
				
				SetList (GetParent (hDlg) ) ;
				dw += dwSize ;
				dwSize = 0 ;
				}
			
			// infinite loop, check for end of block!
			dwSize += dwBlock + 4L ;
			if (!dwBlock)
				break ;
			}
		}
	}

void ImportGetROM ()
	{
	bGetROM = TRUE ;
	
	DialogBox (g_hInstance, MAKEINTRESOURCE (IDD_IMPORTROM), g_hwndMain, TapeImportDlgProc) ;
	
	bGetROM = FALSE ;
	}

// add file
static char		g_szTapeAddFileFile[_MAX_PATH] ;
static TAPEHEADER	*g_ptphdrAddFile ;

static void NEAR TapeAddFileOpen (HWND hwnd)
	{
	char			szBuf[_MAX_PATH] ;
	static const char BASED_CODE	szTitle[] = "Open Add Tape File" ;
	TAPEHEADER	*ptphdr ;
	int			i ;
	OPENFILENAME	ofn ;
	
	szBuf[0] = '\0' ;
	memset (&ofn, 0, sizeof (OPENFILENAME) ) ;
	ofn.lStructSize  = sizeof (OPENFILENAME) ;
	ofn.hwndOwner    = hwnd ;
	ofn.lpstrFilter  = g_szTapeFilter ;
	ofn.lpstrFile    = szBuf ;
	ofn.nMaxFile     = sizeof (szBuf) ;
	ofn.lpstrTitle   = szTitle ;
	ofn.Flags        = OFN_FILEMUSTEXIST | OFN_HIDEREADONLY ;
	ofn.lpstrDefExt  = g_szTapeDefExt ;
	
	if (GetOpenFileName (&ofn) )
		{
		AnsiUpper (szBuf) ;
		
		if (!strcmp (szBuf, g_szTapeFile) )
			{
			MessageID (hwnd, IDS_TAPESAMEIMPORT, MB_ICONEXCLAMATION) ;
			return ;
			}
		
		ptphdr = GetTapeInfo (hwnd, szBuf) ;
		
		if (ptphdr != NULL)
			{
			if (g_szTapeAddFileFile[0])
				FreeLocal ((void**)&g_ptphdrAddFile) ;
			
			g_ptphdrAddFile = ptphdr ;
			
			strcpy (g_szTapeAddFileFile, szBuf) ;
			
			SendDlgItemMessage (hwnd, IDC_RECORDINGSLIST, 
				LB_RESETCONTENT, 0, 0L) ;
			
			i = 0 ;
			while (i < (int)g_ptphdrAddFile->wTapeRecordings)
				{
				wsprintf (szBuf, "%s\t%s", 
					(LPSTR)g_ptphdrAddFile->tapinf[i].szTitle, 
					(LPSTR)g_ptphdrAddFile->tapinf[i].szType) ;
				
				SendDlgItemMessage (hwnd, IDC_RECORDINGSLIST, 
					LB_ADDSTRING, 0, (LPARAM)(LPSTR) szBuf) ;
				i++ ;
				}
			
			EnableWindow (GetDlgItem (hwnd, IDOK), FALSE) ;
			}
		}
	
	return ;
	}

static void NEAR TapeAddFile (HWND hwnd)
	{
	int			i, x, n, nElements ;
	void _huge*	hpData ;
	char			szBuf[256], szBuf2[256] ;
	HCURSOR		hcurOld ;
	BOOL			bExist ;
	DWORD		dw ;
	
	if (!SizeLocal ((void**)&g_ptphdr, 
		(UINT) sizeof (TAPEHEADER) + (g_ptphdr->wTapeRecordings + 
		(int)SendDlgItemMessage (hwnd, IDC_RECORDINGSLIST, 
		LB_GETSELCOUNT, 0, 0L) ) * sizeof (TAPEINFO) ) )
			return ;
	
	hcurOld = SetCursor (LoadCursor (NULL, IDC_WAIT) ) ;
	ShowCursor (TRUE) ;
	
	nElements = (int)SendDlgItemMessage (hwnd, IDC_RECORDINGSLIST, 
			LB_GETCOUNT, 0, 0L) ;
	for (x=0;x<nElements;x++)
		{
		if ( !SendDlgItemMessage (hwnd, IDC_RECORDINGSLIST, 
			LB_GETSEL, x, 0L) )
			continue ;
		
		SendDlgItemMessage (hwnd, IDC_RECORDINGSLIST, 
			LB_GETTEXT, (WPARAM) x, (LPARAM)(LPSTR) szBuf) ;
		i = GetListIndex (szBuf, g_ptphdrAddFile) ;
		
		for (n=0,bExist=FALSE;n<(int)g_ptphdr->wTapeRecordings;n++)
			{
			if (!strcmp (g_ptphdrAddFile->tapinf[i].szTitle, g_ptphdr->tapinf[n].szTitle) && 
				!strcmp (g_ptphdrAddFile->tapinf[i].szType, g_ptphdr->tapinf[n].szType) )
				{
				LoadString (g_hInstance, IDS_TAPEEXIST, szBuf, 256) ;
				
				wsprintf (szBuf2, szBuf,
					(LPSTR)g_ptphdrAddFile->tapinf[i].szTitle, 
					(LPSTR)g_ptphdrAddFile->tapinf[i].szType) ;
				
				MessageBox (hwnd, szBuf2, g_szAppName, MB_OK | MB_ICONEXCLAMATION) ;
				
				bExist = TRUE ;
				break ;
				}
			}
		if (bExist)
			continue ;
		
		hpData = TapeLoadChunk (hwnd, g_szTapeAddFileFile, 
			g_ptphdrAddFile->tapinf[i].fccChunk, &dw) ;
		if (hpData == NULL)
			{
			ShowCursor (FALSE) ;
			SetCursor (hcurOld) ;
			
			break ;
			}
		
		g_ptphdr->tapinf[g_ptphdr->wTapeRecordings] = g_ptphdrAddFile->tapinf[i] ;
		g_ptphdr->tapinf[g_ptphdr->wTapeRecordings].fccChunk = 
			TapeCreateChunk (hwnd, g_szTapeFile, hpData, dw) ;
		if (g_ptphdr->tapinf[g_ptphdr->wTapeRecordings].fccChunk == (FOURCC)0L)
			{
			FreeGlobal ((void FAR**)&hpData) ;	
			
			ShowCursor (FALSE) ;
			SetCursor (hcurOld) ;
			
			break ;
			}
		
		g_ptphdr->wTapeRecordings++ ;
		
		FreeGlobal ((void FAR**)&hpData) ;
		}
	
	SetTapeInfo (hwnd, g_szTapeFile, g_ptphdr) ;
	SetList (GetParent (hwnd) ) ;
	
	ShowCursor (FALSE) ;
	SetCursor (hcurOld) ;
	
	return ;
	}

BOOL FAR PASCAL _export TapeAddFileDlgProc (HWND hDlg, UINT message, UINT wParam, 
												LONG lParam)
	{
	int			i ;
	
	switch (message)
		{
		case WM_INITDIALOG :
			i = 125 ;
			SendDlgItemMessage (hDlg, IDC_RECORDINGSLIST, 
				LB_SETTABSTOPS, 1, (LPARAM)(int FAR*) &i) ;
			g_szTapeAddFileFile[0] = 0 ;
			TapeAddFileOpen (hDlg) ;
			SetFonts (hDlg) ;
			assert (g_szTapeFile[0]) ;
			SetWindowText (GetDlgItem (hDlg, IDC_CURTAPEFILE), g_szTapeFile) ;
			SetWindowText (GetDlgItem (hDlg, IDC_IMPTAPFILE), (g_szTapeAddFileFile[0] ?
				(LPCSTR)g_szTapeAddFileFile : (LPCSTR)szNoAddFile) ) ;
			
			EnableWindow (GetDlgItem (hDlg, IDOK), FALSE) ;
			
			return TRUE ;
		
		case WM_COMMAND :
			switch (wParam)
				{
				case IDC_RECORDINGSLIST:
					if (HIWORD (lParam) == LBN_SELCHANGE)
						{
						EnableWindow (GetDlgItem (hDlg, IDOK), 
							(SendMessage ((HWND)LOWORD (lParam), 
							LB_GETSELCOUNT, 0, 0L) != 0) ) ;
						}
					
					return TRUE ;
				
				case IDOK:
					TapeAddFile (hDlg) ;
					
				case IDCANCEL :
					if (g_szTapeAddFileFile[0])
						FreeLocal ((void**)&g_ptphdrAddFile) ;
					
					EndDialog (hDlg, wParam == IDOK) ;
					
					return TRUE ;
				
				case IDC_FILE:
					TapeAddFileOpen (hDlg) ;
					SetWindowText (GetDlgItem (hDlg, IDC_IMPTAPFILE), (g_szTapeAddFileFile[0] ?
						(LPCSTR)g_szTapeAddFileFile : (LPCSTR)szNoAddFile) ) ;
					
					return TRUE ;
				}
			
			break ;
		}
	
	return FALSE ;
	}

void ExportGetROM ()
	{
	char		szFile[_MAX_PATH] ;
	HFILE	hFile ;
	DWORD	dw ;
	LPBYTE	lpData ;
	static char BASED_CODE szGetROM[] = "GETROM.BIN" ;
	static BYTE BASED_CODE byGetROM[] = {
		0x10, 0x00, 0x00, 0x00, 0xd0, 0xd0, 0xd0, 0xd0, 
		0xd0, 0xd0, 0xd0, 0xd0, 0xd0, 0xd0, 'G' , 'e' , 
		't' , 'R' , 'O' , 'M' } ;
	
	GetModuleFileName (g_hInstance, szFile, sizeof (szFile) ) ;
	
	_fstrcpy (strrchr (szFile, '\\') + 1, szGetROM) ;
	
	hFile = _lopen (szFile, READ) ;
	if (hFile == HFILE_ERROR)
		{
		MessageID (g_hwndMain, IDS_NOGETROM, MB_ICONEXCLAMATION) ;
		
		return ;
		}
	
	dw = _llseek (hFile, 0L, 2) - 1 ;
	
	if (!NewGlobal ((void FAR**)&lpData, GMEM_MOVEABLE, dw + sizeof (byGetROM) + 8) )
		{
		_lclose (hFile) ;
		
		return ;
		}
	
	hpExp = lpData ;
	
	_fmemcpy (lpData, byGetROM, sizeof (byGetROM) ) ;
	lpData += sizeof (byGetROM) ;
	*(DWORD FAR*)lpData = dw ;
	lpData += 4 ;
	_llseek (hFile, 1L, 0) ;
	_hread (hFile, lpData, dw) ;
	_lclose (hFile) ;
	lpData += dw ;
	dw = 0xffffffff ;
	*(DWORD FAR*)lpData = dw ;
	
	DialogBox (g_hInstance, MAKEINTRESOURCE (IDD_TAPEEXPORT), g_hwndMain, TapeExportDlgProc) ;
	
	FreeGlobal (&lpData) ;
	}
